/*
 * Copyright (c) 2008-present Sonatype, Inc.
 *
 * All rights reserved. Includes the third-party code listed at http://links.sonatype.com/products/nexus/pro/attributions
 * Sonatype and Sonatype Nexus are trademarks of Sonatype, Inc. Apache Maven is a trademark of the Apache Foundation.
 * M2Eclipse is a trademark of the Eclipse Foundation. All other trademarks are the property of their respective owners.
 */
/*global Ext, NX*/

/**
 * Health Check info panel.
 *
 * @since 3.0
 */
Ext.define('NX.proui.view.healthcheck.HealthCheckAssetDetail', {
  extend: 'Ext.panel.Panel',
  alias: 'widget.nx-proui-healthcheck-healthcheckassetdetail',
  requires: [
    'NX.util.Base64',
    'NX.util.Url',
    'NX.Windows',
    'NX.I18n'
  ],
  ui: 'nx-inset',

  layout: {
    type: 'vbox',
    align: 'stretch',
    pack: 'start'
  },

  /**
   * Name of repository containing asset to be shown.
   * @private
   */
  repositoryName: undefined,

  /**
   * Format of asset to be shown.
   * @private
   */
  format: undefined,

  /**
   * Path of asset to be shown.
   * @private
   */
  path: undefined,

  /**
   * @private
   * True if info was successfully loaded from server.
   */
  loaded: false,

  /**
   * @private
   * True when active in toolbar.
   */
  active: false,

  listeners: {
    afterrender: function() {
      var me = this;
      me.active = true;
      me.loadInfo();
    }
  },

  /**

   * Shows Health Check information about selected file.
   *
   * @public
   * @param {NX.coreui.model.Asset} assetModel selected asset
   */
  showInfo: function(assetModel) {
    var me = this;
    me.format = assetModel.get('format');
    me.repositoryName = assetModel.get('containingRepositoryName');
    me.path = assetModel.get('name');
    me.loaded = false;
    me.loadInfo();
  },

  /**
   * @private
   * Loads Health Check information about selected file.
   */
  loadInfo: function() {
    var me = this;

    if (!me.loaded) {
      me.removeAll();
      if (me.active && me.rendered && me.path) {
        me.getEl().mask(NX.I18n.get('HealthCheckInfo_Loading_Text'));
        me.add([
          {
            xtype: 'panel',
            ui: 'nx-subsection',
            frame: true,
            minHeight: 600,

            items: [
              {
                xtype: 'box',
                border: false,
                autoEl: {
                  tag: 'iframe',
                  src: NX.util.Url.urlOf('service/rest/healthcheck/healthCheckFileDetail' +
                      '/' + NX.util.Base64.encode(me.path) +
                      '/' + NX.util.Base64.encode(me.repositoryName) +
                      '/index.html?' +
                      Ext.Object.toQueryString({
                        format: me.format
                      })),
                  width: '100%',
                  height: '100%',
                  scrolling: 'auto'
                },
                listeners: {
                  afterrender: function(iframe) {
                    me.loaded = true;
                    iframe.getEl().on({
                      load: function() {
                        var insight = iframe.getEl().dom.contentWindow.Insight;
                        me.getEl().unmask();
                        if (insight) {
                          me.setClmComponent(insight);
                          me.registerClmViewDetailsListener(insight);
                        }
                      }
                    });
                  }
                }
              }
            ]
          }
        ]);
      }
    }
  },

  /**
   * @private
   */
  setClmComponent: function(insight) {
    var me = this;

    NX.direct.healthcheck_AssetDetail.identify(me.repositoryName, me.path, function(response) {
      if (Ext.isObject(response)) {
        if (response.success) {
          var format = response.data.componentIdentifier.format,
              coordinates = response.data.componentIdentifier.coordinates,
              hash = response.data.hash;

          if (me.isLegacyClm(insight)) {
            me.setLegacyClmComponent(insight, format, coordinates, hash);
          }
          else {
            insight.setCoordinates(format, coordinates, {hash: hash});
          }
        }
        else {
          insight.setError({
            errorMessage: 'Unable to identify component: ' + response.message,
            errorCode: 400
          });
        }
      }
      else {
        insight.setError({
          errorMessage: 'Unable to identify component',
          errorCode: 500
        });
      }
    }, undefined, {skipResultCheck: true});
  },

  /**
   * @private
   */
  setLegacyClmComponent: function(insight, format, coordinates, hash) {
    if (format !== 'maven') {
      insight.setError({
        errorMessage: 'Unsupported repository format, please upgrade to Nexus IQ Server 1.13 or greater.',
        errorCode: 400
      });
      return;
    }

    insight.setGav({
      groupId: coordinates.groupId,
      artifactId: coordinates.artifactId,
      version: coordinates.version,
      extension: coordinates.extension,
      classifier: coordinates.classifier,
      hash: hash
    });
  },

  /**
   * @private
   */
  registerClmViewDetailsListener: function(insight) {
    var me = this;
    if (me.isLegacyClm(insight)) {
      me.registerLegacyClmViewDetailsListener(insight);
    }
    else {
      insight.registerCoordsViewDetailsListener(
          function(appId, format, coordinates, hash) {
            var component = {
                  format: format,
                  coordinates: {}
                },
                queryParams = {
                  appId: appId,
                  hash: hash
                };

            if (coordinates) {
              for (var i = 0; i < coordinates.length; i += 2) {
                component.coordinates[coordinates[i]] = coordinates[i + 1];
              }
            }
            queryParams['componentIdentifier'] = JSON.stringify(component);

            if (coordinates) {
              NX.Windows.open(
                  NX.util.Url.urlOf(
                      'viewdetails.html?' +
                      Ext.Object.toQueryString(queryParams)
                  )
              );
            }
          }
      );
    }
  },

  registerLegacyClmViewDetailsListener: function(insight) {
    insight.registerViewDetailsListener(
        function(appId, groupId, artifactId, version, classifier, extension, hash) {
          var queryParams = {
            appId: appId,
            hash: hash,
            groupId: groupId,
            artifactId: artifactId,
            version: version,
            classifier: classifier,
            extension: extension
          };

          NX.Windows.open(
              NX.util.Url.urlOf(
                  'viewdetails.html?' +
                  Ext.Object.toQueryString(queryParams)
              )
          );
        });
  },

  showDetails: function(appId, hash, component, legacy) {
    Sonatype.view.mainTabPanel.addOrShowTab(me.toTabId(appId, hash, component, legacy),
        Nexus.clm.ComponentDetailView, {
          title: 'IQ Server Detail',
          appId: appId,
          hash: hash,
          component: !legacy ? component : null,
          gav: legacy ? component.coordinates : null
        });
  },

  /**
   * @private
   */
  isLegacyClm: function(insight) {
    return !(insight.registerCoordsViewDetailsListener || insight.setCoordinates);
  }

});
