/*
 * Copyright (c) 2008-present Sonatype, Inc.
 *
 * All rights reserved. Includes the third-party code listed at http://links.sonatype.com/products/nexus/pro/attributions
 * Sonatype and Sonatype Nexus are trademarks of Sonatype, Inc. Apache Maven is a trademark of the Apache Foundation.
 * M2Eclipse is a trademark of the Eclipse Foundation. All other trademarks are the property of their respective owners.
 */
/*global Ext, NX*/

/**
 * Manages health check information (security and license alerts) shown in component details view.
 *
 * @since 3.0
 */
Ext.define('NX.proui.controller.HealthCheckInfo', {
  extend: 'NX.app.Controller',
  requires: [
    'NX.Conditions',
    'NX.Permissions',
    'NX.State',
    'NX.util.Url',
    'NX.I18n',
    'NX.coreui.util.HealthCheckUtil'
  ],

  /**
   * @override
   */
  init: function() {
    var me = this;

    me.listen({
      component: {
        'nx-coreui-component-details': {
          healthCheckLoaded: me.renderHealthCheckFields
        },
        'nx-coreui-component-componentinfo': {
          healthCheckLoaded: me.renderComponentInfoHealthCheckFields
        },
        'nx-coreui-component-componentassetinfo': {
          healthCheckLoaded: me.renderComponentInfoHealthCheckFields
        }
      }
    });
  },

  /**
   * Render healthcheck fields.
   *
   * @private
   * @param {Ext.Panel} panel containing health check info section
   * @param {NX.coreui.model.Component} model component model
   */
  renderHealthCheckFields: function(panel, model) {
    var me = this,
        infoPanel,
        info = {};

    if (model) {
      infoPanel = panel.down('#healthCheckInfoPro');
      if (!infoPanel) {
        infoPanel = panel.add({
          xtype: 'nx-info',
          itemId: 'healthCheckInfoPro'
        });
      }
      info[NX.I18n.get('HealthCheckInfo_Security_Alerts_Label')] = me.renderSecurityAlerts(model);
      info[NX.I18n.get('HealthCheckInfo_License_Threat_Label')] = me.renderLicenseThreat(model);
      infoPanel.showInfo(info);
    }
  },

  /**
   * Render healthcheck fields.
   *
   * @private
   * @param {Ext.Panel} panel containing health check info section
   * @param {NX.coreui.model.Component} model component model
   */
  renderComponentInfoHealthCheckFields: function(panel, model) {
    var me = this;

    if (model) {
      panel.setInfo('healthCheckInfoPro', NX.I18n.get('HealthCheckInfo_Security_Alerts_Label'), me.renderSecurityAlerts(model));
      panel.setInfo('healthCheckInfoPro', NX.I18n.get('HealthCheckInfo_License_Threat_Label'), me.renderLicenseThreat(model));
      panel.showInfo();
    }
  },

  /**
   * Render security alerts column.
   *
   * @private
   * @param {NX.coreui.model.Component} model component model
   * @returns {String} rendered value
   */
  renderSecurityAlerts: function(model) {
    var me = this,
        metadata = {},
        result, securityAlerts, critical, severe, moderate,
        buildChiclet = function(clazz, value) {
          return '<span class="nx-clm-chiclet-small ' + clazz + '">' + value + '</span>';
        };

    result = me.renderPreconditions(model, metadata);
    if (!result) {
      securityAlerts = model.get('healthCheckSecurityAlerts');
      critical = model.get('healthCheckCriticalSecurityAlerts');
      severe = model.get('healthCheckSevereSecurityAlerts');
      moderate = model.get('healthCheckModerateSecurityAlerts');
      if (securityAlerts === 0 || securityAlerts > 0) {
        result = buildChiclet(critical > 0 ? 'critical' : '', critical > 0 ? critical : '')
            + buildChiclet(severe > 0 ? 'severe' : '', severe > 0 ? severe : '')
            + buildChiclet(moderate > 0 ? 'moderate' : '', moderate > 0 ? moderate : '');
      }
      if (!Ext.isDefined(result)) {
        result = me.renderNotAvailable(metadata);
      }
    }
    return '<div ' + (metadata.attr || '') + '>' + result + '</div>';
  },

  /**
   * Render license threat column.
   *
   * @private
   * @param {NX.coreui.model.Component} model component model
   * @returns {String} rendered value
   */
  renderLicenseThreat: function(model) {
    var me = this,
        metadata = {},
        result, licenseThreat, licenseThreatName, barClass = '';

    result = me.renderPreconditions(model, metadata);
    if (!result) {
      licenseThreat = model.get('healthCheckLicenseThreat');
      licenseThreatName = model.get('healthCheckLicenseThreatName');
      if (licenseThreat === 0 || licenseThreat > 0) {
        if (licenseThreat === 4) {
          barClass = 'critical';
        }
        else if (licenseThreat === 2) {
          barClass = 'moderate';
        }
        else if (licenseThreat === 1) {
          barClass = 'none';
        }
        else {
          barClass = 'severe';
        }
        return '<span class="nx-clm-chiclet-small ' + barClass + '" style="width: auto; padding: 1px 2px; font-size: inherit;">' + licenseThreatName + '</span>';
      }
      if (!Ext.isDefined(result)) {
        result = me.renderNotAvailable(metadata);
      }
    }
    return '<div ' + (metadata.attr || '') + '>' + result + '</div>';
  },

  /**
   * Render value based on preconditions.
   *
   * @private
   * @param {NX.coreui.model.Component} model component model
   * @param metadata column metadata
   * @returns {*} rendered value
   */
  renderPreconditions: function(model, metadata) {
    var edition = NX.State.getEdition(),
        util = NX.coreui.util.HealthCheckUtil;

    if ('EVAL' === edition || 'OSS' === edition) {
      metadata.attr = 'data-qtip="' + NX.I18n.get('HealthCheckInfo_Tooltip') + '"';
      return util.iconSpan('fa-lock', 'opacity: 0.33;');
    }
    else if (!NX.State.getUser()) {
      metadata.attr = 'data-qtip="' + NX.I18n.get('HealthCheckInfo_LoggedInOnly_Tooltip') + '"';
      return util.iconSpan('fa-lock', 'opacity: 0.33;');
    }
    else if (!NX.Permissions.check('nexus:healthcheckalerts:update')) {
      metadata.attr = 'data-qtip="' + NX.I18n.get('HealthCheckInfo_NoPermission_Tooltip') + '"';
      return util.iconSpan('fa-lock', 'opacity: 0.33;');
    }
    else if (model.get('healthCheckLoading')) {
      return NX.I18n.get('HealthCheckInfo_Loading_Text');
    }
    else if (model.get('healthCheckDisabled')) {
      metadata.attr = 'data-qtip="' + NX.I18n.get('HealthCheckInfo_Disabled_Tooltip') + '"';
      return util.iconSpan('fa-info-circle', 'opacity: 0.33;');
    }
    else if (model.get('healthCheckError')) {
      metadata.attr = 'data-qtip="' + NX.I18n.get('HealthCheckInfo_Error_Tooltip') + '"';
      return util.iconSpan('fa-exclamation-triangle', 'color: red;');
    }
    else if (model.get('healthCheckCapped')
        || (model && model.get('cappedAlerts'))) {
      metadata.attr = 'data-qtip="' + NX.I18n.get('HealthCheckInfo_Quota_Tooltip') + '"';
      return util.iconSpan('fa-exclamation-triangle', 'color: yellow;');
    }
    return undefined;
  },

  /**
   * Render a not available value (no data).
   *
   * @private
   * @param metadata column metadata
   * @returns {String} rendered value
   */
  renderNotAvailable: function(metadata) {
    metadata.attr = 'data-qtip="' + NX.I18n.get('HealthCheckInfo_Unavailable_Tooltip') + '"';
    return NX.coreui.util.HealthCheckUtil.iconSpan('fa-ban', 'opacity: 0.33;');
  }

});
