/*
 * Copyright (c) 2008-present Sonatype, Inc.
 *
 * All rights reserved. Includes the third-party code listed at http://links.sonatype.com/products/nexus/pro/attributions
 * Sonatype and Sonatype Nexus are trademarks of Sonatype, Inc. Apache Maven is a trademark of the Apache Foundation.
 * M2Eclipse is a trademark of the Eclipse Foundation. All other trademarks are the property of their respective owners.
 */
/*global Ext, NX*/

/**
 * Health Check Asset details controller.
 *
 * Shows healthcheck details when asset details are shown.
 *
 * @since 3.0
 */
Ext.define('NX.proui.controller.HealthCheckAssetDetail', {
  extend: 'NX.app.Controller',
  requires: [
    'NX.Permissions',
    'NX.State'
  ],

  views: [
    'healthcheck.HealthCheckAssetDetail'
  ],

  /**
   * @override
   */
  init: function() {
    var me = this;

    me.listen({
      component: {
        'nx-coreui-component-assetcontainer': {
          updated: me.onAssetContainerUpdated
        },
        'nx-coreui-component-componentassetinfo': {
          updated: me.onComponentAssetInfoUpdated
        }
      }
    });
  },

  /**
   * Shows Health Check information about selected asset if user is logged in, has "nexus:healthcheckalerts:update" and
   * we have a PRO edition.
   *
   * @private
   * @param {NX.coreui.view.component.AssetContainer} container asset container
   * @param {NX.coreui.model.Asset} assetModel selected asset
   */
  onAssetContainerUpdated: function(container, asset) {
    var isValid = Boolean((asset.get('componentId') && asset.get('componentId').trim()));
    var panel = container.down('nx-proui-healthcheck-healthcheckassetdetail');

    if (this.isHealthCheckEnabled() && isValid) {
      if (!panel) {
        container.addTab(
            {
              xtype: 'nx-proui-healthcheck-healthcheckassetdetail',
              title: NX.I18n.get('Component_AssetInfo_HealthCheck_Title'),
              itemId: 'healthCheckInfo',
              weight: 50
            }
        );
        panel = container.down('nx-proui-healthcheck-healthcheckassetdetail');
      }

      panel.showInfo(asset);
    }
    else if (panel) {
      container.remove(panel);
    }
  },

  onComponentAssetInfoUpdated: function(container, asset) {
    var isValid = Boolean((asset.get('componentId') && asset.get('componentId').trim()));
    var panel = container.down('nx-proui-healthcheck-healthcheckassetdetail');
    if (this.isHealthCheckEnabled() && isValid) {
      if (!panel) {
        panel = container.add(
          {
            xtype: 'nx-proui-healthcheck-healthcheckassetdetail',
            title: NX.I18n.get('Component_AssetInfo_HealthCheck_Title'),
            itemId: 'healthCheckInfo',
            weight: 30,
            collapsible: true
          }
        );
      }

      panel.showInfo(asset);
    } else if (panel) {
      container.remove(panel);
    }
  },

  isHealthCheckEnabled: function() {
    var edition = NX.State.getEdition();
    return NX.State.getUser() &&
           NX.Permissions.check('nexus:healthcheckalerts:read') &&
           !('EVAL' === edition || 'OSS' === edition);
  }

});
