/*
 * Copyright (c) 2008-present Sonatype, Inc.
 *
 * All rights reserved. Includes the third-party code listed at http://links.sonatype.com/products/nexus/pro/attributions
 * Sonatype and Sonatype Nexus are trademarks of Sonatype, Inc. Apache Maven is a trademark of the Apache Foundation.
 * M2Eclipse is a trademark of the Eclipse Foundation. All other trademarks are the property of their respective owners.
 */
/*global Ext, NX*/

/**
 * Tag List details controller.
 *
 * Shows tags data associated with a selected component or asset.
 *
 * @since 3.14
 */
Ext.define('NX.proui.controller.ComponentTags', {
  extend: 'NX.app.Controller',
  requires: [
    'NX.Permissions',
    'NX.State'
  ],

  views: [
    'tags.ComponentTags'
  ],

  models: [
    'ComponentTags'
  ],

  stores: [
    'ComponentTags'
  ],

  /**
   * @override
   */
  init: function() {
    var me = this;

    me.listen({
      component: {
        'nx-coreui-component-componentassetinfo': {
          updated: me.onComponentAssetInfoUpdated
        },
        'nx-coreui-component-assetcontainer': {
          updated: me.onAssetContainerInfoUpdated
        },
        'nx-coreui-component-componentinfo': {
          updated: me.onComponentInfoUpdated
        }
      }
    });
  },

  /**
   * Shows tag information about selected asset if user is logged in, has "nexus:tags:read" and
   * we have a PRO edition.
   *
   * @private
   * @param {NX.coreui.view.component.AssetContainer} container
   * @param {NX.coreui.model.AssetModel} assetModel selected asset
   */
  onAssetContainerInfoUpdated: function(container, assetModel) {
    var me = this;
    var panel = container.down('nx-proui-tags-taglistdetails');

    if (this.isTaggingEnabled()) {
      if (!panel) {
        // add a tab to the container
        container.addTab(
          {
            xtype: 'nx-proui-tags-taglistdetails',
            ui: 'nx-inset',
            title: NX.I18n.get('ComponentTags_TagTitle'),
            itemId: 'taglistdetails',
            weight: 40,
            autoScroll: true
          }
        );
      }
      me.loadComponentData(assetModel.get('componentId'), assetModel.get('repositoryName'));
    } else {
      container.remove(panel);
    }
  },

  /**
   * Shows tag information about selected asset if user is logged in, has "nexus:tags:read" and
   * we have a PRO edition.
   *
   * @private
   * @param {NX.coreui.view.component.ComponentAssetInfo} container
   * @param {NX.coreui.model.Asset} asset selected asset
   * @param {NX.coreui.model.Component} component selected asset's component
   */
  onComponentAssetInfoUpdated: function(container, asset, component) {
    this.onComponentUpdated(container, component.get('id'), component.get('repositoryName'));
  },

  /**
   * Shows tag information about the selected component if user is logged in, has "nexus:tags:read" and
   * we have a PRO edition.
   *
   * @private
   * @param {NX.coreui.view.component.ComponentAssetInfo} container
   * @param {NX.coreui.model.Component} component selected asset's component
   */
  onComponentInfoUpdated: function(container, component) {
    this.onComponentUpdated(container, component.get('id'), component.get('repositoryName'));
  },

  /**
   * @private
   */
  onComponentUpdated: function(container, componentId, repositoryName) {
    var me = this;
    var panel = container.down('nx-proui-tags-taglistdetails');

    if (this.isTaggingEnabled()) {
      if (!panel) {
        // add a collapsible panel to the container
        container.add(
            {
              xtype: 'nx-proui-tags-taglistdetails',
              ui: 'nx-inset',
              title: NX.I18n.get('ComponentTags_TagTitle'),
              itemId: 'tagPanel',
              weight: 20,
              autoScroll: true,
              collapsible: true
            }
        );
      }
      me.loadComponentData(componentId, repositoryName);
    } else {
      container.remove(panel);
    }
  },

  loadComponentData: function(componentId, repositoryName) {
    var me = this;
    var tagComponentStore = me.getStore('ComponentTags');
    tagComponentStore.load({
      params: {
        filter: [{property: 'componentId',value :componentId}, {property:'repositoryName', value:repositoryName}]
      },
      scope: this
    });
  },

  isTaggingEnabled: function() {
    var edition = NX.State.getEdition();
    return NX.State.getUser() && NX.Permissions.check('nexus:tags:read') &&
        !('EVAL' === edition || 'OSS' === edition);
  }
});
