/*
 * Copyright (c) 2008-present Sonatype, Inc.
 *
 * All rights reserved. Includes the third-party code listed at http://links.sonatype.com/products/nexus/pro/attributions
 * Sonatype and Sonatype Nexus are trademarks of Sonatype, Inc. Apache Maven is a trademark of the Apache Foundation.
 * M2Eclipse is a trademark of the Eclipse Foundation. All other trademarks are the property of their respective owners.
 */
/*global Ext, NX, axios*/

/**
 * SamlAuth controller.
 *
 * @since 3.22
 */
Ext.define('NX.saml.controller.SamlAuth', {
  extend: 'NX.app.Controller',
  requires: [
    'NX.State',
    'NX.saml.view.SamlSignInWindow'
  ],

  listen: {
    controller: {
      '#User': {
        beforeauthenticate: 'authenticate',
        beforereauthenticate: 'reauthenticate',
        authTokenRequest: 'authTokenRequest'
      },
    },
    component: {
      'button[action=signInWithSso]': {
        click: 'doSamlSignin',
      },
      'button[action=signInLocally]': {
        click: 'doLocalSignin'
      }
    }
  },

  refs: [
    {
      ref: 'samlSignInWindow',
      selector: 'nx-saml-signin'
    }
  ],

  views: [
      'SamlSignInWindow'
  ],

  showLocalSignInWindow: undefined,

  /**
   * Handles the beforeauthenticate event
   * @param showLocalSignInWindow - a callback function to call when clicking on the "login without sso" button
   */
  authenticate: function(showLocalSignInWindow) {
    if (!NX.State.getValue('samlEnabled', false)) {
      console.debug('saml not enabled, showing regular login dialog');
      return true;
    }

    this.showLocalSignInWindow = showLocalSignInWindow;
    this.getSamlSignInWindowView().create();

    return false;
  },

  /**
   * Handles the beforereauthenticate event to skip showing the reauthenticate dialog for SAML users
   */
  reauthenticate: function() {
    var authenticatedRealms = NX.State.getUser().authenticatedRealms || [];
    if (Ext.Array.contains(authenticatedRealms, 'SamlRealm')) {
      console.warn('SAML user attempted to access a feature they have no permissions for, skip reauthentication');
      return false;
    }
  },

  /**
   * Add a new authTokenHandler and resolve it from a backend request if the user is a SAML user
   */
  authTokenRequest: function(handlers) {
    var deferred = new Ext.Deferred(),
        authenticatedRealms = NX.State.getUser().authenticatedRealms || [];

    if (Ext.Array.contains(authenticatedRealms, 'SamlRealm')) {
      axios.get('/service/rest/internal/ui/saml/authToken').then(function(response) {
        deferred.resolve(response.data);
      });
      if (Ext.isArray(handlers)) {
        handlers.push(deferred);
      }
    }

    return true;
  },

  doSamlSignin: function() {
    window.location.href = 'saml';
  },

  doLocalSignin: function() {
    this.getSamlSignInWindow().close();
    this.showLocalSignInWindow();
  }
});
