/*
 * Decompiled with CFR 0.152.
 */
package org.sonatype.nexus.ssl;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Lists;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.cert.Certificate;
import java.security.cert.CertificateException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Map;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import javax.net.ssl.KeyManager;
import javax.net.ssl.TrustManager;
import org.sonatype.goodies.common.ComponentSupport;
import org.sonatype.nexus.crypto.CryptoHelper;
import org.sonatype.nexus.ssl.CertificateUtil;
import org.sonatype.nexus.ssl.KeyStoreManager;
import org.sonatype.nexus.ssl.KeyStoreManagerConfiguration;
import org.sonatype.nexus.ssl.KeystoreException;
import org.sonatype.nexus.ssl.internal.ReloadableX509KeyManager;
import org.sonatype.nexus.ssl.internal.ReloadableX509TrustManager;
import org.sonatype.nexus.ssl.internal.geronimo.FileKeystoreInstance;
import org.sonatype.nexus.ssl.internal.geronimo.KeystoreInstance;
import org.sonatype.nexus.ssl.spi.KeyStoreStorage;
import org.sonatype.nexus.ssl.spi.KeyStoreStorageManager;

public class KeyStoreManagerImpl
extends ComponentSupport
implements KeyStoreManager {
    private static final String PRIVATE_KEY_STORE_NAME = "private.ks";
    private static final String TRUSTED_KEY_STORE_NAME = "trusted.ks";
    private static final String DEFAULT00_KEY_ALIAS = "__default00";
    @VisibleForTesting
    static final String PRIVATE_KEY_ALIAS = "identity";
    private final CryptoHelper crypto;
    private final KeyStoreManagerConfiguration config;
    private final KeystoreInstance privateKeyStore;
    private final KeystoreInstance trustedKeyStore;
    private final ReentrantReadWriteLock privateKeyStoreLock = new ReentrantReadWriteLock();
    private final ReentrantReadWriteLock trustedKeyStoreLock = new ReentrantReadWriteLock();
    private ReloadableX509TrustManager reloadableX509TrustManager;
    private ReloadableX509KeyManager reloadableX509KeyManager;

    public KeyStoreManagerImpl(CryptoHelper crypto, KeyStoreStorageManager storageManager, KeyStoreManagerConfiguration config) {
        this.crypto = (CryptoHelper)Preconditions.checkNotNull((Object)crypto);
        this.config = (KeyStoreManagerConfiguration)Preconditions.checkNotNull((Object)config);
        this.privateKeyStore = this.initializePrivateKeyStore(storageManager.createStorage(PRIVATE_KEY_STORE_NAME));
        this.trustedKeyStore = this.initializeTrustedKeyStore(storageManager.createStorage(TRUSTED_KEY_STORE_NAME));
    }

    @VisibleForTesting
    KeyStoreManagerImpl(CryptoHelper crypto, KeyStoreManagerConfiguration config, KeystoreInstance privateKeyStore, KeystoreInstance trustedKeyStore) {
        this.crypto = (CryptoHelper)Preconditions.checkNotNull((Object)crypto);
        this.config = (KeyStoreManagerConfiguration)Preconditions.checkNotNull((Object)config);
        this.privateKeyStore = privateKeyStore;
        this.trustedKeyStore = trustedKeyStore;
    }

    private KeystoreInstance initializePrivateKeyStore(KeyStoreStorage storage) {
        this.log.debug("Initializing private key-store: {}", (Object)storage);
        FileKeystoreInstance ks = new FileKeystoreInstance(this.crypto, storage, PRIVATE_KEY_STORE_NAME, this.config.getPrivateKeyStorePassword(), this.config.getKeyStoreType(), (Map<String, char[]>)ImmutableMap.of((Object)PRIVATE_KEY_ALIAS, (Object)this.config.getPrivateKeyPassword()));
        if (!this.isKeyPairInstalled(ks, DEFAULT00_KEY_ALIAS)) {
            try {
                this.log.debug("Initializing private key-store");
                ks.generateKeyPair(DEFAULT00_KEY_ALIAS, this.config.getPrivateKeyStorePassword(), this.config.getPrivateKeyPassword(), this.config.getKeyAlgorithm(), this.config.getKeyAlgorithmSize(), this.config.getSignatureAlgorithm(), this.config.getCertificateValidity().toDaysI(), DEFAULT00_KEY_ALIAS, "Nexus", "Sonatype", "Silver Spring", "MD", "US");
                Certificate cert = ks.getCertificate(DEFAULT00_KEY_ALIAS, this.config.getPrivateKeyStorePassword());
                this.log.trace("Generated default certificate:\n{}", (Object)cert);
            }
            catch (KeystoreException e) {
                this.log.error("Failed to install default certificate", (Throwable)e);
                throw new RuntimeException(e);
            }
        }
        if (this.log.isTraceEnabled()) {
            try {
                String[] aliases = ks.listPrivateKeys(this.config.getPrivateKeyStorePassword());
                if (aliases != null && aliases.length != 0) {
                    this.log.trace("Private key aliases:");
                    String[] stringArray = aliases;
                    int n = aliases.length;
                    int n2 = 0;
                    while (n2 < n) {
                        String alias = stringArray[n2];
                        this.log.trace("  {}", (Object)alias);
                        ++n2;
                    }
                }
            }
            catch (KeystoreException e) {
                this.log.error("Failed to list key aliases", (Throwable)e);
            }
        }
        this.log.debug("Private key-store initialized");
        return ks;
    }

    private KeystoreInstance initializeTrustedKeyStore(KeyStoreStorage storage) {
        this.log.debug("Initializing trusted key-store: {}", (Object)storage);
        FileKeystoreInstance ks = new FileKeystoreInstance(this.crypto, storage, TRUSTED_KEY_STORE_NAME, this.config.getTrustedKeyStorePassword(), this.config.getKeyStoreType(), (Map<String, char[]>)ImmutableMap.of((Object)TRUSTED_KEY_STORE_NAME, (Object)this.config.getTrustedKeyStorePassword()));
        this.logTrustedCertificateAliases(ks);
        try {
            if (ks.listTrustCertificates(this.config.getTrustedKeyStorePassword()).length != 0) {
                this.log.warn("Trusted key-store should have been empty when initialized but was not");
            }
        }
        catch (KeystoreException keystoreException) {}
        this.log.debug("Trusted key-store initialized");
        return ks;
    }

    private void logTrustedCertificateAliases(KeystoreInstance ks) {
        Preconditions.checkNotNull((Object)ks);
        if (this.log.isTraceEnabled()) {
            try {
                String[] aliases = ks.listTrustCertificates(this.config.getTrustedKeyStorePassword());
                if (aliases != null && aliases.length != 0) {
                    this.log.trace("Trusted certificate aliases:");
                    String[] stringArray = aliases;
                    int n = aliases.length;
                    int n2 = 0;
                    while (n2 < n) {
                        String alias = stringArray[n2];
                        this.log.trace("  {}", (Object)alias);
                        ++n2;
                    }
                }
            }
            catch (KeystoreException e) {
                this.log.error("Failed to list aliases", (Throwable)e);
            }
        }
    }

    @Override
    public TrustManager[] getTrustManagers() throws KeystoreException {
        this.trustedKeyStoreLock.readLock().lock();
        try {
            TrustManager[] trustManagers = this.trustedKeyStore.getTrustManager(this.config.getTrustManagerAlgorithm(), this.config.getTrustedKeyStorePassword());
            try {
                this.reloadableX509TrustManager = ReloadableX509TrustManager.replaceX509TrustManager(this.reloadableX509TrustManager, trustManagers);
            }
            catch (NoSuchAlgorithmException e) {
                throw new KeystoreException("A ReloadableX509TrustManager could not be created.", e);
            }
            TrustManager[] trustManagerArray = trustManagers;
            return trustManagerArray;
        }
        finally {
            this.trustedKeyStoreLock.readLock().unlock();
        }
    }

    @Override
    public KeyManager[] getKeyManagers() throws KeystoreException {
        this.privateKeyStoreLock.readLock().lock();
        try {
            KeyManager[] keyManagers = this.privateKeyStore.getKeyManager(this.config.getKeyManagerAlgorithm(), PRIVATE_KEY_ALIAS, this.config.getPrivateKeyStorePassword());
            try {
                this.reloadableX509KeyManager = ReloadableX509KeyManager.replaceX509KeyManager(this.reloadableX509KeyManager, keyManagers);
            }
            catch (NoSuchAlgorithmException e) {
                throw new KeystoreException("A ReloadableX509KeyManager could not be created.", e);
            }
            KeyManager[] keyManagerArray = keyManagers;
            return keyManagerArray;
        }
        finally {
            this.privateKeyStoreLock.readLock().unlock();
        }
    }

    @Override
    public void importTrustCertificate(Certificate certificate, String alias) throws KeystoreException {
        this.trustedKeyStoreLock.writeLock().lock();
        try {
            this.log.debug("Importing trust certificate w/alias: {}", (Object)alias);
            if (this.trustedKeyStore.getCertificate(alias) != null) {
                this.log.warn("Certificate already exists in trust-store w/alias: {}; replacing certificate", (Object)alias);
                this.trustedKeyStore.deleteEntry(alias, this.config.getTrustedKeyStorePassword());
            }
            this.trustedKeyStore.importTrustCertificate(certificate, alias, this.config.getTrustedKeyStorePassword());
            this.logTrustedCertificateAliases(this.trustedKeyStore);
            this.getTrustManagers();
        }
        finally {
            this.trustedKeyStoreLock.writeLock().unlock();
        }
    }

    @Override
    public void importTrustCertificate(String certificateInPEM, String alias) throws KeystoreException, CertificateException {
        Certificate certificate = CertificateUtil.decodePEMFormattedCertificate(certificateInPEM);
        this.importTrustCertificate(certificate, alias);
    }

    @Override
    public Certificate getTrustedCertificate(String alias) throws KeystoreException {
        this.trustedKeyStoreLock.readLock().lock();
        try {
            Certificate certificate = this.trustedKeyStore.getCertificate((String)Preconditions.checkNotNull((Object)alias, (Object)"'alias' cannot be null when looking up a trusted Certificate."), this.config.getTrustedKeyStorePassword());
            return certificate;
        }
        finally {
            this.trustedKeyStoreLock.readLock().unlock();
        }
    }

    @Override
    public Collection<Certificate> getTrustedCertificates() throws KeystoreException {
        this.trustedKeyStoreLock.readLock().lock();
        try {
            String[] aliases = this.trustedKeyStore.listTrustCertificates(this.config.getTrustedKeyStorePassword());
            ArrayList certificates = Lists.newArrayListWithCapacity((int)aliases.length);
            String[] stringArray = aliases;
            int n = aliases.length;
            int n2 = 0;
            while (n2 < n) {
                String alias = stringArray[n2];
                Certificate cert = this.trustedKeyStore.getCertificate(alias);
                if (cert == null) {
                    this.log.warn("Trust-store reports it contains certificate for alias '{}' but certificate is null", (Object)alias);
                } else {
                    certificates.add(cert);
                }
                ++n2;
            }
            ArrayList arrayList = certificates;
            return arrayList;
        }
        finally {
            this.trustedKeyStoreLock.readLock().unlock();
        }
    }

    @Override
    public void removeTrustCertificate(String alias) throws KeystoreException {
        this.trustedKeyStoreLock.writeLock().lock();
        try {
            this.log.debug("Removing trust certificate w/alias: {}", (Object)alias);
            this.trustedKeyStore.deleteEntry(alias, this.config.getTrustedKeyStorePassword());
            this.logTrustedCertificateAliases(this.trustedKeyStore);
            this.getTrustManagers();
        }
        finally {
            this.trustedKeyStoreLock.writeLock().unlock();
        }
    }

    @Override
    public void generateAndStoreKeyPair(String commonName, String organizationalUnit, String organization, String locality, String state, String country) throws KeystoreException {
        this.privateKeyStoreLock.writeLock().lock();
        try {
            this.privateKeyStore.generateKeyPair(PRIVATE_KEY_ALIAS, this.config.getPrivateKeyStorePassword(), this.config.getPrivateKeyPassword(), this.config.getKeyAlgorithm(), this.config.getKeyAlgorithmSize(), this.config.getSignatureAlgorithm(), this.config.getCertificateValidity().toDaysI(), commonName, organizationalUnit, organization, locality, state, country);
            this.getKeyManagers();
        }
        finally {
            this.privateKeyStoreLock.writeLock().unlock();
        }
    }

    private boolean isKeyPairInstalled(KeystoreInstance ks, String alias) {
        try {
            ks.getCertificate(alias, this.config.getPrivateKeyStorePassword());
            return true;
        }
        catch (KeystoreException keystoreException) {
            this.log.trace("Key-pair not installed w/alias: {}", (Object)alias);
            return false;
        }
    }

    @Override
    public boolean isKeyPairInitialized() {
        this.privateKeyStoreLock.readLock().lock();
        try {
            boolean bl = this.isKeyPairInstalled(this.privateKeyStore, PRIVATE_KEY_ALIAS);
            return bl;
        }
        finally {
            this.privateKeyStoreLock.readLock().unlock();
        }
    }

    @Override
    public Certificate getCertificate() throws KeystoreException {
        this.privateKeyStoreLock.readLock().lock();
        try {
            Certificate certificate = this.privateKeyStore.getCertificate(PRIVATE_KEY_ALIAS, this.config.getPrivateKeyStorePassword());
            return certificate;
        }
        finally {
            this.privateKeyStoreLock.readLock().unlock();
        }
    }

    @Override
    public PrivateKey getPrivateKey() throws KeystoreException {
        this.privateKeyStoreLock.readLock().lock();
        try {
            PrivateKey privateKey = this.privateKeyStore.getPrivateKey(PRIVATE_KEY_ALIAS, this.config.getPrivateKeyStorePassword(), this.config.getPrivateKeyPassword());
            return privateKey;
        }
        finally {
            this.privateKeyStoreLock.readLock().unlock();
        }
    }

    @Override
    public void removePrivateKey() throws KeystoreException {
        this.privateKeyStoreLock.writeLock().lock();
        try {
            this.privateKeyStore.deleteEntry(PRIVATE_KEY_ALIAS, this.config.getPrivateKeyStorePassword());
        }
        finally {
            this.privateKeyStoreLock.writeLock().unlock();
        }
    }

    @Override
    public void reloadTrustedKeystore() throws KeystoreException {
        this.trustedKeyStore.reloadIfReplicating(this.config.getTrustedKeyStorePassword());
    }
}

