/*
 * Decompiled with CFR 0.152.
 */
package org.sonatype.nexus.common.io;

import com.google.common.base.Function;
import com.google.common.base.Preconditions;
import com.google.common.base.Predicate;
import java.io.File;
import java.io.IOException;
import java.nio.file.DirectoryStream;
import java.nio.file.FileAlreadyExistsException;
import java.nio.file.FileStore;
import java.nio.file.FileVisitOption;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.util.EnumSet;
import java.util.Set;
import java.util.concurrent.atomic.AtomicInteger;
import javax.annotation.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class DirectoryHelper {
    private static final Logger log = LoggerFactory.getLogger(DirectoryHelper.class);
    public static final Set<FileVisitOption> DEFAULT_FILE_VISIT_OPTIONS = EnumSet.of(FileVisitOption.FOLLOW_LINKS);

    private DirectoryHelper() {
    }

    public static void mkdir(Path dir) throws IOException {
        block2: {
            try {
                Files.createDirectories(dir, new FileAttribute[0]);
            }
            catch (FileAlreadyExistsException e) {
                if (Files.isDirectory(dir, new LinkOption[0])) break block2;
                throw e;
            }
        }
    }

    public static void mkdir(File dir) throws IOException {
        DirectoryHelper.mkdir(dir.toPath());
    }

    public static File mkdir(File parent, String child) throws IOException {
        File dir = new File(parent, child);
        DirectoryHelper.mkdir(dir);
        return dir;
    }

    public static void clean(Path dir) throws IOException {
        DirectoryHelper.validateDirectory(dir);
        Files.walkFileTree(dir, DEFAULT_FILE_VISIT_OPTIONS, Integer.MAX_VALUE, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

            @Override
            public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                Files.delete(file);
                return FileVisitResult.CONTINUE;
            }
        });
    }

    public static boolean cleanIfExists(Path dir) throws IOException {
        Preconditions.checkNotNull((Object)dir);
        if (Files.exists(dir, new LinkOption[0])) {
            DirectoryHelper.clean(dir);
            return true;
        }
        return false;
    }

    public static void empty(final Path dir) throws IOException {
        DirectoryHelper.validateDirectory(dir);
        Files.walkFileTree(dir, DEFAULT_FILE_VISIT_OPTIONS, Integer.MAX_VALUE, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

            @Override
            public FileVisitResult visitFile(Path f, BasicFileAttributes attrs) throws IOException {
                Files.deleteIfExists(f);
                return FileVisitResult.CONTINUE;
            }

            @Override
            public FileVisitResult postVisitDirectory(Path d, IOException exc) throws IOException {
                if (exc != null) {
                    throw exc;
                }
                if (dir != d) {
                    Files.deleteIfExists(d);
                }
                return FileVisitResult.CONTINUE;
            }
        });
    }

    public static int deleteIfEmptyRecursively(Path dir, final Long timestamp) throws IOException {
        final AtomicInteger deleteCount = new AtomicInteger(0);
        File rootDir = dir.toFile();
        if (!rootDir.exists()) {
            log.debug("Requested path {} doesn't exist, will not process for empty directories to remove.", (Object)rootDir.getAbsolutePath());
            return 0;
        }
        Files.walkFileTree(dir, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

            @Override
            public FileVisitResult postVisitDirectory(Path dir, IOException exc) throws IOException {
                File dirFile = dir.toFile();
                if (!dirFile.exists()) {
                    log.debug("Processing directory {} that no longer exists, will ignore and move on", (Object)dir.toAbsolutePath());
                    return FileVisitResult.CONTINUE;
                }
                if (timestamp != null && dirFile.lastModified() > timestamp) {
                    log.debug("Processing directory {} has been modified recently and will not be removed.", (Object)dir.toAbsolutePath());
                    return FileVisitResult.CONTINUE;
                }
                String[] items = dir.toFile().list();
                if (items != null && items.length == 0) {
                    try {
                        Files.delete(dir);
                        deleteCount.incrementAndGet();
                    }
                    catch (IOException e) {
                        log.error("Failed to delete empty directory {} will stop processing.", (Object)dir.toAbsolutePath(), (Object)e);
                        return FileVisitResult.TERMINATE;
                    }
                }
                return FileVisitResult.CONTINUE;
            }
        });
        return deleteCount.intValue();
    }

    public static boolean emptyIfExists(Path dir) throws IOException {
        Preconditions.checkNotNull((Object)dir);
        if (Files.exists(dir, new LinkOption[0])) {
            DirectoryHelper.empty(dir);
            return true;
        }
        return false;
    }

    public static void delete(Path dir) throws IOException {
        DirectoryHelper.delete(dir, null);
    }

    public static void delete(Path dir, final @Nullable Predicate<Path> excludeFilter) throws IOException {
        DirectoryHelper.validateDirectoryOrFile(dir);
        if (Files.isDirectory(dir, new LinkOption[0])) {
            Files.walkFileTree(dir, DEFAULT_FILE_VISIT_OPTIONS, Integer.MAX_VALUE, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

                @Override
                public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) throws IOException {
                    if (excludeFilter != null && excludeFilter.apply((Object)dir)) {
                        return FileVisitResult.SKIP_SUBTREE;
                    }
                    return FileVisitResult.CONTINUE;
                }

                @Override
                public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                    Files.delete(file);
                    return FileVisitResult.CONTINUE;
                }

                @Override
                public FileVisitResult postVisitDirectory(Path dir, IOException exc) throws IOException {
                    if (exc != null) {
                        throw exc;
                    }
                    boolean needsDelete = true;
                    if (excludeFilter != null) {
                        Throwable throwable = null;
                        Object var5_6 = null;
                        try (DirectoryStream<Path> dirStream = Files.newDirectoryStream(dir);){
                            if (dirStream.iterator().hasNext()) {
                                needsDelete = false;
                            }
                        }
                        catch (Throwable throwable2) {
                            if (throwable == null) {
                                throwable = throwable2;
                            } else if (throwable != throwable2) {
                                throwable.addSuppressed(throwable2);
                            }
                            throw throwable;
                        }
                    }
                    if (needsDelete) {
                        Files.delete(dir);
                    }
                    return FileVisitResult.CONTINUE;
                }
            });
        } else {
            Files.delete(dir);
        }
    }

    public static boolean deleteIfExists(Path dir) throws IOException {
        return DirectoryHelper.deleteIfExists(dir, null);
    }

    public static boolean deleteIfExists(Path dir, @Nullable Predicate<Path> excludeFilter) throws IOException {
        Preconditions.checkNotNull((Object)dir);
        if (Files.exists(dir, new LinkOption[0])) {
            DirectoryHelper.delete(dir, excludeFilter);
            return true;
        }
        return false;
    }

    public static void copy(Path from, Path to) throws IOException {
        DirectoryHelper.copy(from, to, null);
    }

    public static void copy(Path from, Path to, @Nullable Predicate<Path> excludeFilter) throws IOException {
        DirectoryHelper.validateDirectoryOrFile(from);
        Preconditions.checkNotNull((Object)to);
        if (Files.isDirectory(from, new LinkOption[0])) {
            Preconditions.checkArgument((!DirectoryHelper.isParentOf(from, to) || excludeFilter != null ? 1 : 0) != 0);
            Files.walkFileTree(from, DEFAULT_FILE_VISIT_OPTIONS, Integer.MAX_VALUE, new CopyVisitor(from, to, excludeFilter));
        } else {
            DirectoryHelper.mkdir(to.getParent());
            Files.copy(from, to, StandardCopyOption.REPLACE_EXISTING);
        }
    }

    public static boolean copyIfExists(Path from, Path to) throws IOException {
        return DirectoryHelper.copyIfExists(from, to, null);
    }

    public static boolean copyIfExists(Path from, Path to, @Nullable Predicate<Path> excludeFilter) throws IOException {
        Preconditions.checkNotNull((Object)from);
        if (Files.exists(from, new LinkOption[0])) {
            DirectoryHelper.copy(from, to, excludeFilter);
            return true;
        }
        return false;
    }

    private static void sameFileStoreMove(Path from, Path to) throws IOException {
        Files.move(from, to, StandardCopyOption.REPLACE_EXISTING);
    }

    private static void crossFileStoreMove(Path from, Path to) throws IOException {
        DirectoryHelper.copyDeleteMove(from, to, null);
    }

    private static boolean areOnSameFileStore(Path from, Path to) {
        try {
            FileStore fromStore = Files.getFileStore(from);
            Path toExistingParent = to.normalize();
            while (toExistingParent != null && !Files.exists(toExistingParent, new LinkOption[0])) {
                toExistingParent = toExistingParent.getParent();
            }
            if (toExistingParent != null) {
                FileStore toStore = Files.getFileStore(toExistingParent);
                return fromStore.equals(toStore);
            }
            log.warn("No ultimate parent path found for '{}'", (Object)to, (Object)new RuntimeException("marker"));
            return false;
        }
        catch (IOException iOException) {
            return false;
        }
    }

    public static void move(Path from, Path to) throws IOException {
        if (DirectoryHelper.areOnSameFileStore(from, to)) {
            DirectoryHelper.sameFileStoreMove(from, to);
        } else {
            DirectoryHelper.crossFileStoreMove(from, to);
        }
    }

    public static boolean moveIfExists(Path from, Path to) throws IOException {
        Preconditions.checkNotNull((Object)from);
        if (Files.exists(from, new LinkOption[0])) {
            DirectoryHelper.move(from, to);
            return true;
        }
        return false;
    }

    public static void copyDeleteMove(Path from, Path to, @Nullable Predicate<Path> excludeFilter) throws IOException {
        DirectoryHelper.copy(from, to, excludeFilter);
        DirectoryHelper.delete(from, excludeFilter);
    }

    public static boolean copyDeleteMoveIfExists(Path from, Path to, @Nullable Predicate<Path> excludeFilter) throws IOException {
        Preconditions.checkNotNull((Object)from);
        if (Files.exists(from, new LinkOption[0])) {
            DirectoryHelper.copyDeleteMove(from, to, excludeFilter);
            return true;
        }
        return false;
    }

    public static void apply(Path from, Function<Path, FileVisitResult> func) throws IOException {
        DirectoryHelper.validateDirectory(from);
        Files.walkFileTree(from, DEFAULT_FILE_VISIT_OPTIONS, Integer.MAX_VALUE, new FunctionVisitor(func));
    }

    public static void applyToFiles(Path from, Function<Path, FileVisitResult> func) throws IOException {
        DirectoryHelper.validateDirectory(from);
        Files.walkFileTree(from, DEFAULT_FILE_VISIT_OPTIONS, Integer.MAX_VALUE, new FunctionFileVisitor(func));
    }

    private static void validateDirectory(Path ... paths) {
        Path[] pathArray = paths;
        int n = paths.length;
        int n2 = 0;
        while (n2 < n) {
            Path path = pathArray[n2];
            Preconditions.checkNotNull((Object)path, (Object)"Path must be non-null");
            Preconditions.checkArgument((boolean)Files.isDirectory(path, new LinkOption[0]), (String)"%s is not a directory", (Object)path);
            ++n2;
        }
    }

    private static void validateDirectoryOrFile(Path ... paths) {
        Path[] pathArray = paths;
        int n = paths.length;
        int n2 = 0;
        while (n2 < n) {
            Path path = pathArray[n2];
            Preconditions.checkNotNull((Object)path, (Object)"Path must be non-null");
            Preconditions.checkArgument((boolean)Files.exists(path, new LinkOption[0]), (String)"%s does not exists", (Object)path);
            ++n2;
        }
    }

    private static boolean isParentOf(Path possibleParent, Path possibleChild) {
        return possibleChild.startsWith(possibleParent);
    }

    private static class CopyVisitor
    extends SimpleFileVisitor<Path> {
        private final Path from;
        private final Path to;
        private final Predicate<Path> excludeFilter;

        public CopyVisitor(Path from, Path to, @Nullable Predicate<Path> excludeFilter) {
            this.from = from;
            this.to = to;
            this.excludeFilter = excludeFilter;
        }

        @Override
        public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes a) throws IOException {
            if (this.excludeFilter != null && this.excludeFilter.apply((Object)dir)) {
                return FileVisitResult.SKIP_SUBTREE;
            }
            Path targetPath = this.to.resolve(this.from.relativize(dir));
            if (!Files.exists(targetPath, new LinkOption[0])) {
                DirectoryHelper.mkdir(targetPath);
            }
            return FileVisitResult.CONTINUE;
        }

        @Override
        public FileVisitResult visitFile(Path file, BasicFileAttributes a) throws IOException {
            Files.copy(file, this.to.resolve(this.from.relativize(file)), StandardCopyOption.REPLACE_EXISTING);
            return FileVisitResult.CONTINUE;
        }
    }

    private static class FunctionFileVisitor
    extends SimpleFileVisitor<Path> {
        private final Function<Path, FileVisitResult> func;

        public FunctionFileVisitor(Function<Path, FileVisitResult> func) {
            this.func = (Function)Preconditions.checkNotNull(func);
        }

        @Override
        public FileVisitResult visitFile(Path file, BasicFileAttributes a) throws IOException {
            return (FileVisitResult)((Object)this.func.apply((Object)file));
        }
    }

    private static class FunctionVisitor
    extends SimpleFileVisitor<Path> {
        private final Function<Path, FileVisitResult> func;

        public FunctionVisitor(Function<Path, FileVisitResult> func) {
            this.func = (Function)Preconditions.checkNotNull(func);
        }

        @Override
        public FileVisitResult visitFile(Path file, BasicFileAttributes a) throws IOException {
            return (FileVisitResult)((Object)this.func.apply((Object)file));
        }

        @Override
        public FileVisitResult postVisitDirectory(Path dir, IOException e) throws IOException {
            if (e != null) {
                throw e;
            }
            return (FileVisitResult)((Object)this.func.apply((Object)dir));
        }
    }
}

