/*
 * Decompiled with CFR 0.152.
 */
package org.sonatype.nexus.repository.tools.datastore;

import com.google.common.base.Preconditions;
import com.google.common.base.Stopwatch;
import java.io.IOException;
import java.io.InputStream;
import java.time.OffsetDateTime;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import javax.inject.Inject;
import javax.inject.Named;
import javax.validation.constraints.NotNull;
import org.sonatype.goodies.common.ComponentSupport;
import org.sonatype.nexus.blobstore.api.Blob;
import org.sonatype.nexus.blobstore.api.BlobMetrics;
import org.sonatype.nexus.blobstore.api.BlobStoreException;
import org.sonatype.nexus.blobstore.api.BlobStoreManager;
import org.sonatype.nexus.common.entity.Continuation;
import org.sonatype.nexus.common.entity.Continuations;
import org.sonatype.nexus.common.hash.HashAlgorithm;
import org.sonatype.nexus.repository.Repository;
import org.sonatype.nexus.repository.content.Asset;
import org.sonatype.nexus.repository.content.AssetBlob;
import org.sonatype.nexus.repository.content.RepositoryContent;
import org.sonatype.nexus.repository.content.facet.ContentFacet;
import org.sonatype.nexus.repository.content.fluent.FluentAsset;
import org.sonatype.nexus.repository.content.fluent.FluentAssets;
import org.sonatype.nexus.repository.tools.BlobUnavilableException;
import org.sonatype.nexus.repository.tools.DeadBlobFinder;
import org.sonatype.nexus.repository.tools.DeadBlobResult;
import org.sonatype.nexus.repository.tools.MismatchedSHA1Exception;
import org.sonatype.nexus.repository.tools.ResultState;

@Named
public class DatastoreDeadBlobFinder
extends ComponentSupport
implements DeadBlobFinder<Asset> {
    private BlobStoreManager blobStoreManager;

    @Inject
    public DatastoreDeadBlobFinder(BlobStoreManager blobStoreManager) {
        this.blobStoreManager = blobStoreManager;
    }

    public void findAndProcessBatch(@NotNull Repository repository, boolean ignoreMissingBlobRefs, int batchSize, Consumer<DeadBlobResult<Asset>> resultProcessor) {
        Preconditions.checkNotNull((Object)repository);
        Preconditions.checkNotNull(resultProcessor);
        FluentAssets fluentAssets = ((ContentFacet)repository.facet(ContentFacet.class)).assets();
        Continuation assets = fluentAssets.browse(batchSize, null);
        long deadBlobCandidateCount = 0L;
        long deadBlobCount = 0L;
        Stopwatch sw = Stopwatch.createStarted();
        while (!assets.isEmpty()) {
            List<DeadBlobResult<Asset>> deadBlobCandidates = this.identifySuspects(repository, ignoreMissingBlobRefs, assets.stream(), true);
            List<DeadBlobResult<Asset>> deadBlobResults = this.verifySuspects(deadBlobCandidates, repository, true);
            deadBlobCandidateCount += (long)deadBlobCandidates.size();
            deadBlobCount += (long)deadBlobResults.size();
            for (DeadBlobResult<Asset> deadBlobResult : deadBlobResults) {
                resultProcessor.accept(deadBlobResult);
            }
            assets = fluentAssets.browse(batchSize, assets.nextContinuationToken());
        }
        this.log.info("Inspection of repository {} took {}ms for {} assets and identified {} incorrect Assets", new Object[]{repository.getName(), sw.elapsed(TimeUnit.MILLISECONDS), deadBlobCandidateCount, deadBlobCount});
    }

    public List<DeadBlobResult<Asset>> find(@NotNull Repository repository, boolean ignoreMissingBlobRefs) {
        Preconditions.checkNotNull((Object)repository);
        FluentAssets fluentAssets = ((ContentFacet)repository.facet(ContentFacet.class)).assets();
        List<DeadBlobResult<Asset>> deadBlobCandidates = this.identifySuspects(repository, ignoreMissingBlobRefs, Continuations.streamOf(fluentAssets::browse), false);
        return this.verifySuspects(deadBlobCandidates, repository, false);
    }

    private List<DeadBlobResult<Asset>> identifySuspects(Repository repository, boolean ignoreMissingBlobRefs, Stream<FluentAsset> fluentAssets, boolean batchMode) {
        Stopwatch sw = Stopwatch.createStarted();
        AtomicLong blobsExamined = new AtomicLong();
        List<DeadBlobResult<Asset>> deadBlobCandidates = fluentAssets.peek(a -> {
            long l = blobsExamined.incrementAndGet();
        }).map(asset -> {
            if (!asset.blob().isPresent() && ignoreMissingBlobRefs) {
                this.log.trace("Set to ignore missing blobRef on {}", asset);
                return null;
            }
            return this.checkAsset(repository.getName(), (Asset)asset);
        }).filter(Objects::nonNull).collect(Collectors.toList());
        if (!batchMode) {
            this.log.debug("Inspecting repository {} took {}ms for {}  assets and identified {} potentially incorrect Assets for followup assessment", new Object[]{repository.getName(), sw.elapsed(TimeUnit.MILLISECONDS), blobsExamined, deadBlobCandidates.size()});
        }
        return deadBlobCandidates;
    }

    private List<DeadBlobResult<Asset>> verifySuspects(List<DeadBlobResult<Asset>> deadBlobCandidates, Repository repository, boolean batchMode) {
        if (!deadBlobCandidates.isEmpty()) {
            Stopwatch sw = Stopwatch.createStarted();
            ContentFacet content = (ContentFacet)repository.facet(ContentFacet.class);
            List<DeadBlobResult<Asset>> deadBlobs = deadBlobCandidates.stream().map(candidateResult -> {
                DeadBlobResult<Asset> deadBlobResult = this.checkAsset(repository.getName(), content.assets().path(((Asset)candidateResult.getAsset()).path()).find().orElse(null));
                if (deadBlobResult != null) {
                    this.logResults((DeadBlobResult<Asset>)candidateResult, deadBlobResult);
                    return deadBlobResult;
                }
                this.log.debug("Asset {} corrected from error state {} during inspection", (Object)((Asset)candidateResult.getAsset()).path(), (Object)candidateResult.getResultState());
                return null;
            }).filter(Objects::nonNull).collect(Collectors.toList());
            if (!batchMode) {
                this.log.info("Followup inspection of repository {} took {}ms for {} assets and identified {} incorrect Assets", new Object[]{repository.getName(), sw.elapsed(TimeUnit.MILLISECONDS), deadBlobCandidates.size(), deadBlobs.size()});
            }
            return deadBlobs;
        }
        return Collections.emptyList();
    }

    private DeadBlobResult<Asset> checkAsset(String repositoryName, Asset asset) {
        if (asset == null) {
            return new DeadBlobResult(repositoryName, null, ResultState.ASSET_DELETED, "Asset was deleted during inspection");
        }
        try {
            Blob blob = asset.blob().map(AssetBlob::blobRef).map(blobRef -> this.blobStoreManager.get(blobRef.getStore()).get(blobRef.getBlobId())).orElseThrow(() -> new IllegalStateException("Blob not found."));
            this.verifyBlob(blob, asset);
        }
        catch (IllegalStateException ise) {
            return new DeadBlobResult(repositoryName, (Object)asset, ResultState.MISSING_BLOB_REF, ise.getMessage());
        }
        catch (BlobStoreException bse) {
            if (bse.getCause() instanceof IOException) {
                return new DeadBlobResult(repositoryName, (Object)asset, ResultState.UNREADABLE_BLOB, bse.getMessage());
            }
            return new DeadBlobResult(repositoryName, (Object)asset, ResultState.DELETED, bse.getMessage());
        }
        catch (MismatchedSHA1Exception pae) {
            return new DeadBlobResult(repositoryName, (Object)asset, ResultState.SHA1_DISAGREEMENT, pae.getMessage());
        }
        catch (BlobUnavilableException e) {
            return new DeadBlobResult(repositoryName, (Object)asset, ResultState.UNAVAILABLE_BLOB, e.getMessage() == null ? "Blob inputstream unavailable" : e.getMessage());
        }
        catch (Exception e) {
            return new DeadBlobResult(repositoryName, (Object)asset, ResultState.UNKNOWN, e.getMessage());
        }
        return null;
    }

    private void verifyBlob(Blob blob, Asset asset) throws MismatchedSHA1Exception, BlobUnavilableException, IOException {
        BlobMetrics metrics = blob.getMetrics();
        String assetChecksum = asset.blob().map(AssetBlob::checksums).map(checksums -> (String)checksums.get(HashAlgorithm.SHA1.name())).orElse(null);
        if (!metrics.getSha1Hash().equals(assetChecksum)) {
            throw new MismatchedSHA1Exception();
        }
        Throwable throwable = null;
        Object var6_7 = null;
        try (InputStream blobstream = blob.getInputStream();){
            if (metrics.getContentSize() > 0L && blobstream.available() == 0) {
                throw new BlobUnavilableException();
            }
        }
        catch (Throwable throwable2) {
            if (throwable == null) {
                throwable = throwable2;
            } else if (throwable != throwable2) {
                throwable.addSuppressed(throwable2);
            }
            throw throwable;
        }
    }

    private void logResults(DeadBlobResult<Asset> firstResult, DeadBlobResult<Asset> secondResult) {
        this.log.info("Possible bad data found in Asset: {}", secondResult.getAsset());
        if (DatastoreDeadBlobFinder.lastUpdated(firstResult) != DatastoreDeadBlobFinder.lastUpdated(secondResult)) {
            this.log.info("Asset metadata was updated during inspection between {} and {}", (Object)DatastoreDeadBlobFinder.lastUpdated(firstResult), (Object)DatastoreDeadBlobFinder.lastUpdated(secondResult));
        }
        if (firstResult.getResultState() != secondResult.getResultState()) {
            this.log.info("Error state changed from {} to {} during inspection", (Object)firstResult.getResultState(), (Object)secondResult.getResultState());
        }
        if (DatastoreDeadBlobFinder.blobUpdated(firstResult) != DatastoreDeadBlobFinder.blobUpdated(secondResult)) {
            this.log.info("Asset blob was updated during inspection between {} and {}", (Object)DatastoreDeadBlobFinder.blobUpdated(firstResult), (Object)DatastoreDeadBlobFinder.blobUpdated(secondResult));
        }
    }

    private static OffsetDateTime blobUpdated(DeadBlobResult<Asset> deadBlobResult) {
        return Optional.ofNullable((Asset)deadBlobResult.getAsset()).flatMap(Asset::blob).map(AssetBlob::blobCreated).orElse(null);
    }

    private static OffsetDateTime lastUpdated(DeadBlobResult<Asset> deadBlobResult) {
        return Optional.ofNullable((Asset)deadBlobResult.getAsset()).map(RepositoryContent::lastUpdated).orElse(null);
    }
}

