/*
 * Decompiled with CFR 0.152.
 */
package org.sonatype.nexus.repository.storage;

import com.google.common.base.Preconditions;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import com.google.common.hash.HashCode;
import java.io.InputStream;
import java.nio.file.Path;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import javax.annotation.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.sonatype.nexus.blobstore.api.Blob;
import org.sonatype.nexus.blobstore.api.BlobId;
import org.sonatype.nexus.blobstore.api.BlobRef;
import org.sonatype.nexus.blobstore.api.BlobStore;
import org.sonatype.nexus.common.collect.NestedAttributesMap;
import org.sonatype.nexus.common.hash.HashAlgorithm;
import org.sonatype.nexus.common.hash.MultiHashingInputStream;
import org.sonatype.nexus.common.node.NodeAccess;
import org.sonatype.nexus.common.text.Strings2;
import org.sonatype.nexus.repository.storage.AssetBlob;
import org.sonatype.nexus.repository.storage.BlobMetadataStorage;
import org.sonatype.nexus.repository.storage.MissingBlobException;
import org.sonatype.nexus.repository.storage.PrefetchedAssetBlob;

public class BlobTx {
    private static final Logger log = LoggerFactory.getLogger(BlobTx.class);
    private final NodeAccess nodeAccess;
    private final BlobStore blobStore;
    private final BlobMetadataStorage blobMetadataStorage;
    private final Set<AssetBlob> newlyCreatedBlobs = Sets.newHashSet();
    private final Map<BlobRef, String> deletionRequests = Maps.newHashMap();

    public BlobTx(NodeAccess nodeAccess, BlobStore blobStore, BlobMetadataStorage blobMetadataStorage) {
        this.nodeAccess = (NodeAccess)Preconditions.checkNotNull((Object)nodeAccess);
        this.blobStore = (BlobStore)Preconditions.checkNotNull((Object)blobStore);
        this.blobMetadataStorage = (BlobMetadataStorage)Preconditions.checkNotNull((Object)blobMetadataStorage);
    }

    public AssetBlob create(InputStream inputStream, Map<String, String> headers, Iterable<HashAlgorithm> hashAlgorithms, String contentType) {
        MultiHashingInputStream hashingStream = new MultiHashingInputStream(hashAlgorithms, inputStream);
        Blob streamedBlob = this.blobStore.create((InputStream)hashingStream, headers);
        return this.createPrefetchedAssetBlob(streamedBlob, hashingStream.hashes(), true, contentType);
    }

    public AssetBlob createByHardLinking(Path sourceFile, Map<String, String> headers, Map<HashAlgorithm, HashCode> hashes, String contentType, long size) {
        return this.createAssetBlob(store -> store.create(sourceFile, headers, size, (HashCode)hashes.get(HashAlgorithm.SHA1)), hashes, false, contentType);
    }

    public AssetBlob createByCopying(BlobRef blobRef, Map<String, String> headers, Map<HashAlgorithm, HashCode> hashes, boolean hashesVerified) {
        Preconditions.checkArgument((!Strings2.isBlank((String)headers.get("BlobStore.content-type")) ? 1 : 0) != 0, (Object)"Blob content type is required");
        if (!blobRef.getStore().equals(this.blobStore.getBlobStoreConfiguration().getName())) {
            throw new MissingBlobException(blobRef);
        }
        return this.createAssetBlob(store -> store.makeBlobPermanent(blobRef.getBlobId(), headers), hashes, hashesVerified, headers.get("BlobStore.content-type"));
    }

    public void attachAssetMetadata(BlobId blobId, NestedAttributesMap componentAttributes, NestedAttributesMap assetAttributes) {
        this.blobMetadataStorage.attach(this.blobStore, blobId, componentAttributes, assetAttributes, null);
    }

    private PrefetchedAssetBlob createPrefetchedAssetBlob(Blob blob, Map<HashAlgorithm, HashCode> hashes, boolean hashesVerified, String contentType) {
        PrefetchedAssetBlob assetBlob = new PrefetchedAssetBlob(this.nodeAccess, this.blobStore, blob, contentType, hashes, hashesVerified);
        this.newlyCreatedBlobs.add(assetBlob);
        return assetBlob;
    }

    private AssetBlob createAssetBlob(Function<BlobStore, Blob> blobFunction, Map<HashAlgorithm, HashCode> hashes, boolean hashesVerified, String contentType) {
        AssetBlob assetBlob = new AssetBlob(this.nodeAccess, this.blobStore, blobFunction, contentType, hashes, hashesVerified);
        this.newlyCreatedBlobs.add(assetBlob);
        return assetBlob;
    }

    @Nullable
    public Blob get(BlobRef blobRef) {
        return this.blobStore.get(blobRef.getBlobId());
    }

    public void delete(BlobRef blobRef, String reason) {
        this.deletionRequests.put(blobRef, reason);
    }

    public void commit() {
        for (Map.Entry<BlobRef, String> deletionRequestEntry : this.deletionRequests.entrySet()) {
            try {
                BlobId blobId = deletionRequestEntry.getKey().getBlobId();
                this.logGAMetadataBlobDetailsIfDebug(this.blobStore.get(blobId), deletionRequestEntry.getValue());
                this.blobStore.delete(blobId, deletionRequestEntry.getValue());
            }
            catch (Throwable t) {
                log.warn("Unable to delete old blob {} while committing transaction", (Object)deletionRequestEntry.getKey(), (Object)t);
            }
        }
        for (AssetBlob assetBlob : this.newlyCreatedBlobs) {
            try {
                if (assetBlob.isAttached()) continue;
                this.logGAMetadataBlobDetailsIfDebug(assetBlob, "Removing unattached asset.");
                assetBlob.delete("Removing unattached asset");
            }
            catch (Throwable t) {
                log.warn("Unable to delete new orphan blob {} while committing transaction", (Object)assetBlob.getBlobRef(), (Object)t);
            }
        }
        this.clearState();
    }

    public void end() {
        for (AssetBlob assetBlob : this.newlyCreatedBlobs) {
            try {
                if (assetBlob.isAttached()) continue;
                this.logGAMetadataBlobDetailsIfDebug(assetBlob, "Rolling back new asset.");
                assetBlob.delete("Rolling back new asset", true);
            }
            catch (Throwable t) {
                log.warn("Unable to delete new blob {} while rolling back transaction", (Object)assetBlob.getBlobRef(), (Object)t);
            }
        }
        this.clearState();
    }

    public void rollback() {
        for (AssetBlob assetBlob : this.newlyCreatedBlobs) {
            assetBlob.setAttached(false);
            this.logGAMetadataBlobDetailsIfDebug(assetBlob, "Unattached blob from asset.");
        }
    }

    private void clearState() {
        this.newlyCreatedBlobs.clear();
        this.deletionRequests.clear();
    }

    private void logGAMetadataBlobDetailsIfDebug(AssetBlob assetBlob, String message) {
        if (log.isDebugEnabled()) {
            Blob blob = assetBlob.getBlob();
            Map headers = blob.getHeaders();
            String blobName = headers.getOrDefault("BlobStore.blob-name", "");
            String repositoryName = headers.getOrDefault("Bucket.repo-name", "");
            if (blobName.endsWith("metadata.properties") && !blobName.endsWith("-SNAPSHOT/maven-metadata.xml")) {
                log.debug("{} blob id {}, blob name {} repository {}.", new Object[]{message, blob.getId(), blobName, repositoryName});
            }
        }
    }

    private void logGAMetadataBlobDetailsIfDebug(Blob blob, String message) {
        Map headers;
        String blobName;
        if (log.isDebugEnabled() && (blobName = (headers = blob.getHeaders()).getOrDefault("BlobStore.blob-name", "")).endsWith("metadata.properties") && !blobName.endsWith("-SNAPSHOT/maven-metadata.xml")) {
            log.debug("Deleting blob - {} blob id {}, blob name {}.", new Object[]{message, blob.getId(), blobName});
        }
    }
}

