/*
 * Decompiled with CFR 0.152.
 */
package org.sonatype.nexus.blobstore.s3.internal;

import com.amazonaws.services.s3.AmazonS3;
import com.amazonaws.services.s3.model.AmazonS3Exception;
import com.amazonaws.services.s3.model.BucketLifecycleConfiguration;
import com.amazonaws.services.s3.model.ListObjectsRequest;
import com.amazonaws.services.s3.model.ObjectListing;
import com.amazonaws.services.s3.model.lifecycle.LifecycleAndOperator;
import com.amazonaws.services.s3.model.lifecycle.LifecycleFilter;
import com.amazonaws.services.s3.model.lifecycle.LifecycleFilterPredicate;
import com.amazonaws.services.s3.model.lifecycle.LifecyclePrefixPredicate;
import com.amazonaws.services.s3.model.lifecycle.LifecycleTagPredicate;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;
import javax.inject.Inject;
import javax.inject.Named;
import org.apache.commons.collections.CollectionUtils;
import org.sonatype.goodies.common.ComponentSupport;
import org.sonatype.nexus.blobstore.StorageLocationManager;
import org.sonatype.nexus.blobstore.api.BlobStoreConfiguration;
import org.sonatype.nexus.blobstore.s3.internal.BucketOwnershipCheckFeatureFlag;
import org.sonatype.nexus.blobstore.s3.internal.S3BlobStore;
import org.sonatype.nexus.blobstore.s3.internal.S3BlobStoreConfigurationHelper;
import org.sonatype.nexus.blobstore.s3.internal.S3BlobStoreException;

@Named
public class BucketManager
extends ComponentSupport
implements StorageLocationManager {
    static final String OLD_LIFECYCLE_EXPIRATION_RULE_ID = "Expire soft-deleted blobstore objects";
    static final String LIFECYCLE_EXPIRATION_RULE_ID_PREFIX = "Expire soft-deleted objects in blobstore ";
    private AmazonS3 s3;
    private final BucketOwnershipCheckFeatureFlag ownershipCheckFeatureFlag;

    @Inject
    public BucketManager(BucketOwnershipCheckFeatureFlag featureFlag) {
        this.ownershipCheckFeatureFlag = (BucketOwnershipCheckFeatureFlag)Preconditions.checkNotNull((Object)featureFlag);
    }

    public void setS3(AmazonS3 s3) {
        this.s3 = s3;
    }

    public void prepareStorageLocation(BlobStoreConfiguration blobStoreConfiguration) {
        String bucket = S3BlobStoreConfigurationHelper.getConfiguredBucket(blobStoreConfiguration);
        this.checkPermissions(S3BlobStoreConfigurationHelper.getConfiguredBucket(blobStoreConfiguration));
        if (!this.s3.doesBucketExistV2(bucket)) {
            try {
                this.s3.createBucket(bucket);
            }
            catch (AmazonS3Exception e) {
                if ("AccessDenied".equals(e.getErrorCode())) {
                    this.log.debug("Error creating bucket {}", (Object)bucket, (Object)e);
                    throw S3BlobStoreException.insufficientCreatePermissionsError();
                }
                this.log.info("Error creating bucket {}", (Object)bucket, (Object)e);
                throw S3BlobStoreException.unexpectedError("creating bucket");
            }
            this.setBucketLifecycleConfiguration(this.s3, blobStoreConfiguration, null);
        } else {
            BucketLifecycleConfiguration lifecycleConfiguration = this.s3.getBucketLifecycleConfiguration(bucket);
            if (!this.isExpirationLifecycleConfigurationPresent(lifecycleConfiguration, blobStoreConfiguration)) {
                this.setBucketLifecycleConfiguration(this.s3, blobStoreConfiguration, lifecycleConfiguration);
            }
        }
    }

    public void deleteStorageLocation(BlobStoreConfiguration blobStoreConfiguration) {
        String bucket = S3BlobStoreConfigurationHelper.getConfiguredBucket(blobStoreConfiguration);
        ObjectListing listing = this.s3.listObjects(new ListObjectsRequest().withBucketName(bucket).withMaxKeys(Integer.valueOf(1)));
        if (listing.getObjectSummaries().isEmpty()) {
            this.s3.deleteBucket(bucket);
        } else {
            this.log.info("Not removing S3 bucket {} because it is not empty", (Object)bucket);
            BucketLifecycleConfiguration lifecycleConfiguration = this.s3.getBucketLifecycleConfiguration(bucket);
            List<BucketLifecycleConfiguration.Rule> nonBlobstoreRules = this.nonBlobstoreRules(lifecycleConfiguration, blobStoreConfiguration.getName());
            if (!CollectionUtils.isEmpty(nonBlobstoreRules)) {
                lifecycleConfiguration.setRules(nonBlobstoreRules);
                this.s3.setBucketLifecycleConfiguration(bucket, lifecycleConfiguration);
            } else {
                this.s3.deleteBucketLifecycleConfiguration(bucket);
            }
        }
    }

    @VisibleForTesting
    boolean isExpirationLifecycleConfigurationPresent(BucketLifecycleConfiguration lifecycleConfiguration, BlobStoreConfiguration blobStoreConfiguration) {
        String bucketPrefix = S3BlobStoreConfigurationHelper.getBucketPrefix(blobStoreConfiguration);
        int expirationInDays = S3BlobStoreConfigurationHelper.getConfiguredExpirationInDays(blobStoreConfiguration);
        return lifecycleConfiguration != null && lifecycleConfiguration.getRules() != null && lifecycleConfiguration.getRules().stream().filter(r -> r.getExpirationInDays() == expirationInDays).anyMatch(r -> this.isDeletedTagPredicate(r.getFilter().getPredicate(), bucketPrefix));
    }

    private BucketLifecycleConfiguration makeLifecycleConfiguration(BucketLifecycleConfiguration existing, BlobStoreConfiguration blobStoreConfiguration) {
        String blobStoreName = blobStoreConfiguration.getName();
        String bucketPrefix = S3BlobStoreConfigurationHelper.getBucketPrefix(blobStoreConfiguration);
        int expirationInDays = S3BlobStoreConfigurationHelper.getConfiguredExpirationInDays(blobStoreConfiguration);
        Object filterPredicate = bucketPrefix.isEmpty() ? new LifecycleTagPredicate(S3BlobStore.DELETED_TAG) : new LifecycleAndOperator(Arrays.asList(new LifecyclePrefixPredicate(bucketPrefix), new LifecycleTagPredicate(S3BlobStore.DELETED_TAG)));
        BucketLifecycleConfiguration.Rule rule = new BucketLifecycleConfiguration.Rule().withId(LIFECYCLE_EXPIRATION_RULE_ID_PREFIX + blobStoreName).withFilter(new LifecycleFilter((LifecycleFilterPredicate)filterPredicate)).withExpirationInDays(expirationInDays).withStatus("Enabled");
        BucketLifecycleConfiguration newConfiguration = null;
        if (existing != null && !existing.getRules().isEmpty()) {
            List<BucketLifecycleConfiguration.Rule> rules = this.nonBlobstoreRules(existing, blobStoreName);
            if (expirationInDays > 0) {
                rules.add(rule);
            }
            if (!rules.isEmpty()) {
                existing.setRules(rules);
                newConfiguration = existing;
            }
        } else if (expirationInDays > 0) {
            newConfiguration = new BucketLifecycleConfiguration().withRules(new BucketLifecycleConfiguration.Rule[]{rule});
        }
        return newConfiguration;
    }

    private List<BucketLifecycleConfiguration.Rule> nonBlobstoreRules(BucketLifecycleConfiguration existing, String blobStoreName) {
        List rules = existing.getRules();
        if (rules == null) {
            return Collections.emptyList();
        }
        return rules.stream().filter(r -> !r.getId().equals(LIFECYCLE_EXPIRATION_RULE_ID_PREFIX + blobStoreName) && !r.getId().equals(OLD_LIFECYCLE_EXPIRATION_RULE_ID)).collect(Collectors.toList());
    }

    private void setBucketLifecycleConfiguration(AmazonS3 s3, BlobStoreConfiguration blobStoreConfiguration, BucketLifecycleConfiguration lifecycleConfiguration) {
        String bucket = S3BlobStoreConfigurationHelper.getConfiguredBucket(blobStoreConfiguration);
        BucketLifecycleConfiguration newLifecycleConfiguration = this.makeLifecycleConfiguration(lifecycleConfiguration, blobStoreConfiguration);
        if (newLifecycleConfiguration != null) {
            s3.setBucketLifecycleConfiguration(bucket, newLifecycleConfiguration);
        } else if (lifecycleConfiguration != null && !lifecycleConfiguration.getRules().isEmpty()) {
            s3.deleteBucketLifecycleConfiguration(bucket);
        }
    }

    private boolean isDeletedTagPredicate(LifecycleFilterPredicate filterPredicate, String bucketPrefix) {
        if (filterPredicate instanceof LifecycleTagPredicate) {
            LifecycleTagPredicate tagPredicate = (LifecycleTagPredicate)filterPredicate;
            return S3BlobStore.DELETED_TAG.equals((Object)tagPredicate.getTag());
        }
        if (filterPredicate instanceof LifecycleAndOperator) {
            LifecycleAndOperator andOperator = (LifecycleAndOperator)filterPredicate;
            return andOperator.getOperands().stream().anyMatch(op -> this.isDeletedTagPredicate((LifecycleFilterPredicate)op, bucketPrefix)) && andOperator.getOperands().stream().anyMatch(op -> this.isBucketPrefixPredicate((LifecycleFilterPredicate)op, bucketPrefix));
        }
        return false;
    }

    private boolean isBucketPrefixPredicate(LifecycleFilterPredicate filterPredicate, String bucketPrefix) {
        if (filterPredicate instanceof LifecyclePrefixPredicate) {
            LifecyclePrefixPredicate prefixPredicate = (LifecyclePrefixPredicate)filterPredicate;
            return prefixPredicate.getPrefix().equals(bucketPrefix);
        }
        return false;
    }

    private void checkPermissions(String bucket) {
        this.checkCredentials(bucket);
        if (!this.ownershipCheckFeatureFlag.isDisabled() && this.s3.doesBucketExistV2(bucket)) {
            this.checkBucketOwner(bucket);
        }
    }

    private void checkCredentials(String bucket) {
        try {
            this.s3.doesBucketExistV2(bucket);
        }
        catch (AmazonS3Exception e) {
            if ("InvalidAccessKeyId".equals(e.getErrorCode()) || "SignatureDoesNotMatch".equals(e.getErrorCode())) {
                this.log.debug("Exception thrown checking AWS credentials", (Throwable)e);
                throw S3BlobStoreException.buildException(e);
            }
            this.log.info("Exception thrown checking AWS credentials.", (Throwable)e);
            throw S3BlobStoreException.unexpectedError("checking credentials");
        }
    }

    private void checkBucketOwner(String bucket) {
        try {
            this.s3.getBucketPolicy(bucket);
        }
        catch (AmazonS3Exception e) {
            String errorCode = e.getErrorCode();
            String logMessage = String.format("Exception thrown checking ownership of \"%s\" bucket.", bucket);
            if ("AccessDenied".equals(errorCode)) {
                this.log.debug(logMessage, (Throwable)e);
                throw S3BlobStoreException.bucketOwnershipError();
            }
            if ("MethodNotAllowed".equals(errorCode)) {
                this.log.debug(logMessage, (Throwable)e);
                throw S3BlobStoreException.invalidIdentityError();
            }
            this.log.info(logMessage, this.log.isDebugEnabled() ? e : e.getMessage());
            throw S3BlobStoreException.unexpectedError("checking bucket ownership");
        }
    }
}

