/*
 * Decompiled with CFR 0.152.
 */
package org.sonatype.nexus.repository.internal.blobstore;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Maps;
import com.google.common.eventbus.Subscribe;
import java.io.InputStream;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BooleanSupplier;
import javax.annotation.Nullable;
import javax.inject.Inject;
import javax.inject.Named;
import javax.inject.Provider;
import javax.inject.Singleton;
import org.sonatype.nexus.blobstore.BlobStoreDescriptor;
import org.sonatype.nexus.blobstore.api.Blob;
import org.sonatype.nexus.blobstore.api.BlobAttributes;
import org.sonatype.nexus.blobstore.api.BlobId;
import org.sonatype.nexus.blobstore.api.BlobInputStreamException;
import org.sonatype.nexus.blobstore.api.BlobSession;
import org.sonatype.nexus.blobstore.api.BlobSessionSupplier;
import org.sonatype.nexus.blobstore.api.BlobStore;
import org.sonatype.nexus.blobstore.api.BlobStoreConfiguration;
import org.sonatype.nexus.blobstore.api.BlobStoreCreatedEvent;
import org.sonatype.nexus.blobstore.api.BlobStoreDeletedEvent;
import org.sonatype.nexus.blobstore.api.BlobStoreException;
import org.sonatype.nexus.blobstore.api.BlobStoreManager;
import org.sonatype.nexus.blobstore.api.BlobStoreNotFoundException;
import org.sonatype.nexus.blobstore.api.BlobStoreStartedEvent;
import org.sonatype.nexus.blobstore.api.BlobStoreStoppedEvent;
import org.sonatype.nexus.blobstore.api.BlobStoreUpdatedEvent;
import org.sonatype.nexus.blobstore.api.DefaultBlobStoreProvider;
import org.sonatype.nexus.blobstore.api.tasks.BlobStoreTaskService;
import org.sonatype.nexus.common.app.FreezeService;
import org.sonatype.nexus.common.event.EventAware;
import org.sonatype.nexus.common.event.EventConsumer;
import org.sonatype.nexus.common.event.EventHelper;
import org.sonatype.nexus.common.event.EventManager;
import org.sonatype.nexus.common.node.NodeAccess;
import org.sonatype.nexus.common.stateguard.Guarded;
import org.sonatype.nexus.common.stateguard.StateGuardLifecycleSupport;
import org.sonatype.nexus.distributed.event.service.api.EventType;
import org.sonatype.nexus.distributed.event.service.api.common.BlobStoreDistributedConfigurationEvent;
import org.sonatype.nexus.jmx.reflect.ManagedObject;
import org.sonatype.nexus.repository.blobstore.BlobStoreConfigurationStore;
import org.sonatype.nexus.repository.internal.blobstore.BlobStoreConfigurationCreatedEvent;
import org.sonatype.nexus.repository.internal.blobstore.BlobStoreConfigurationDeletedEvent;
import org.sonatype.nexus.repository.internal.blobstore.BlobStoreConfigurationEvent;
import org.sonatype.nexus.repository.internal.blobstore.BlobStoreConfigurationUpdatedEvent;
import org.sonatype.nexus.repository.internal.blobstore.BlobStoreOverride;
import org.sonatype.nexus.repository.manager.RepositoryManager;
import org.sonatype.nexus.repository.replication.ReplicationBlobStoreStatusManager;

@Named
@Singleton
@ManagedObject
public class BlobStoreManagerImpl
extends StateGuardLifecycleSupport
implements BlobStoreManager,
BlobSessionSupplier,
EventAware {
    private final EventManager eventManager;
    private final Map<String, BlobStore> stores = Maps.newConcurrentMap();
    private final BlobStoreConfigurationStore store;
    private final Map<String, BlobStoreDescriptor> blobStoreDescriptors;
    private final Map<String, Provider<BlobStore>> blobStorePrototypes;
    private final FreezeService freezeService;
    private final BooleanSupplier provisionDefaults;
    private final Provider<RepositoryManager> repositoryManagerProvider;
    private final BlobStoreTaskService blobStoreTaskService;
    private final Provider<BlobStoreOverride> blobStoreOverrideProvider;
    private final ReplicationBlobStoreStatusManager replicationBlobStoreStatusManager;
    private final DefaultBlobStoreProvider defaultBlobstoreProvider;

    @Inject
    public BlobStoreManagerImpl(EventManager eventManager, BlobStoreConfigurationStore store, Map<String, BlobStoreDescriptor> blobStoreDescriptors, Map<String, Provider<BlobStore>> blobStorePrototypes, FreezeService freezeService, Provider<RepositoryManager> repositoryManagerProvider, NodeAccess nodeAccess, @Nullable @Named(value="${nexus.blobstore.provisionDefaults}") @Nullable @Named(value="${nexus.blobstore.provisionDefaults}") Boolean provisionDefaults, DefaultBlobStoreProvider defaultBlobstoreProvider, BlobStoreTaskService blobStoreTaskService, Provider<BlobStoreOverride> blobStoreOverrideProvider, ReplicationBlobStoreStatusManager replicationBlobStoreStatusManager) {
        this.eventManager = (EventManager)Preconditions.checkNotNull((Object)eventManager);
        this.store = (BlobStoreConfigurationStore)Preconditions.checkNotNull((Object)store);
        this.blobStoreDescriptors = (Map)Preconditions.checkNotNull(blobStoreDescriptors);
        this.blobStorePrototypes = (Map)Preconditions.checkNotNull(blobStorePrototypes);
        this.freezeService = (FreezeService)Preconditions.checkNotNull((Object)freezeService);
        this.repositoryManagerProvider = (Provider)Preconditions.checkNotNull(repositoryManagerProvider);
        this.blobStoreTaskService = (BlobStoreTaskService)Preconditions.checkNotNull((Object)blobStoreTaskService);
        this.blobStoreOverrideProvider = blobStoreOverrideProvider;
        this.replicationBlobStoreStatusManager = (ReplicationBlobStoreStatusManager)Preconditions.checkNotNull((Object)replicationBlobStoreStatusManager);
        this.defaultBlobstoreProvider = (DefaultBlobStoreProvider)Preconditions.checkNotNull((Object)defaultBlobstoreProvider);
        this.provisionDefaults = provisionDefaults != null ? provisionDefaults::booleanValue : () -> !nodeAccess.isClustered();
    }

    protected void doStart() throws Exception {
        Optional.ofNullable((BlobStoreOverride)this.blobStoreOverrideProvider.get()).ifPresent(BlobStoreOverride::apply);
        List configurations = this.store.list();
        if (configurations.isEmpty() && this.provisionDefaults.getAsBoolean()) {
            this.log.debug("No BlobStores configured; provisioning default BlobStore");
            this.store.create(this.defaultBlobstoreProvider.get(this::newConfiguration));
            configurations = this.store.list();
        }
        this.log.debug("Restoring {} BlobStores", (Object)configurations.size());
        for (BlobStoreConfiguration blobStoreConfiguration : configurations) {
            this.log.debug("Restoring BlobStore: {}", (Object)blobStoreConfiguration);
            BlobStore blobStore = null;
            try {
                try {
                    blobStore = (BlobStore)this.blobStorePrototypes.get(blobStoreConfiguration.getType()).get();
                    blobStore.init(blobStoreConfiguration);
                }
                catch (Exception e) {
                    this.log.error("Unable to restore BlobStore {}", (Object)blobStoreConfiguration, (Object)e);
                    this.track(blobStoreConfiguration.getName(), blobStore);
                    continue;
                }
            }
            catch (Throwable throwable) {
                this.track(blobStoreConfiguration.getName(), blobStore);
                throw throwable;
            }
            this.track(blobStoreConfiguration.getName(), blobStore);
        }
        this.log.debug("Starting {} BlobStores", (Object)this.stores.size());
        for (Map.Entry entry : this.stores.entrySet()) {
            String name = (String)entry.getKey();
            BlobStore blobStore = (BlobStore)entry.getValue();
            this.log.debug("Starting BlobStore: {}", (Object)name);
            try {
                blobStore.start();
                this.eventManager.post((Object)new BlobStoreStartedEvent(blobStore));
            }
            catch (Exception e) {
                this.log.error("Unable to start BlobStore {}", (Object)name, (Object)e);
            }
        }
    }

    protected void doStop() throws Exception {
        if (this.stores.isEmpty()) {
            this.log.debug("No BlobStores defined");
            return;
        }
        this.log.debug("Stopping {} BlobStores", (Object)this.stores.size());
        for (Map.Entry<String, BlobStore> entry : this.stores.entrySet()) {
            String name = entry.getKey();
            BlobStore store = entry.getValue();
            this.log.debug("Stopping blob-store: {}", (Object)name);
            store.stop();
            this.eventManager.post((Object)new BlobStoreStoppedEvent(store));
        }
        this.stores.clear();
    }

    @Guarded(by={"STARTED"})
    public Iterable<BlobStore> browse() {
        return ImmutableList.copyOf(this.stores.values());
    }

    @Guarded(by={"STARTED"})
    public BlobStore create(BlobStoreConfiguration configuration) throws Exception {
        Preconditions.checkNotNull((Object)configuration);
        this.log.debug("Creating BlobStore: {} with attributes: {}", (Object)configuration.getName(), (Object)configuration.getAttributes());
        this.validateConfiguration(configuration, true);
        BlobStore blobStore = this.getBlobStoreForCreate(configuration);
        if (!EventHelper.isReplicating()) {
            try {
                this.store.create(configuration);
                this.log.debug("BlobStore: {} created.", (Object)configuration.getName());
            }
            catch (Exception e) {
                try {
                    blobStore.remove();
                }
                catch (Exception removeException) {
                    this.log.error("Error removing BlobStore {} after create failed", (Object)configuration.getName(), (Object)removeException);
                }
                throw e;
            }
        }
        this.doCreate(blobStore, configuration);
        this.log.debug("BlobStore: {} saved into local cache", (Object)configuration.getName());
        this.eventManager.post((Object)new BlobStoreDistributedConfigurationEvent(configuration.getName(), EventType.CREATED));
        return blobStore;
    }

    private BlobStore getBlobStoreForCreate(BlobStoreConfiguration configuration) throws Exception {
        BlobStore blobStore = (BlobStore)this.blobStorePrototypes.get(configuration.getType()).get();
        blobStore.init(configuration);
        this.replicationBlobStoreStatusManager.initializeReplicationStatus(configuration);
        blobStore.validateCanCreateAndUpdate();
        return blobStore;
    }

    private void doCreate(BlobStore blobStore, BlobStoreConfiguration configuration) throws Exception {
        Preconditions.checkNotNull((Object)blobStore, (Object)"blobStore param is NULL");
        Preconditions.checkNotNull((Object)configuration, (Object)"configuration param is NULL");
        String blobStoreName = configuration.getName();
        if (!this.stores.containsKey(blobStoreName)) {
            this.track(blobStoreName, blobStore);
            blobStore.start();
            this.eventManager.post((Object)new BlobStoreCreatedEvent(blobStore));
            this.eventManager.post((Object)new BlobStoreStartedEvent(blobStore));
        }
    }

    @Guarded(by={"STARTED"})
    public BlobStore update(BlobStoreConfiguration configuration) throws Exception {
        Preconditions.checkNotNull((Object)configuration);
        this.validateConfiguration(configuration, false);
        BlobStore blobStore = this.getBlobStoreForUpdate(configuration);
        if (!EventHelper.isReplicating()) {
            this.store.update(configuration);
        }
        this.doUpdate(blobStore, configuration);
        this.eventManager.post((Object)new BlobStoreDistributedConfigurationEvent(configuration.getName(), EventType.UPDATED));
        return blobStore;
    }

    private BlobStore getBlobStoreForUpdate(BlobStoreConfiguration configuration) throws Exception {
        BlobStore blobStore = this.get(configuration.getName());
        Preconditions.checkNotNull((Object)blobStore);
        blobStore.validateCanCreateAndUpdate();
        this.log.debug("Updating BlobStore: {} with attributes: {}", (Object)configuration.getName(), (Object)configuration.getAttributes());
        return blobStore;
    }

    private void doUpdate(BlobStore blobStore, BlobStoreConfiguration configuration) throws Exception {
        BlobStoreConfiguration currentBlobStoreConfiguration = blobStore.getBlobStoreConfiguration();
        if (blobStore.isStarted()) {
            blobStore.stop();
            this.eventManager.post((Object)new BlobStoreStoppedEvent(blobStore));
        }
        try {
            blobStore.init(configuration);
            blobStore.start();
            this.eventManager.post((Object)new BlobStoreUpdatedEvent(blobStore));
            this.eventManager.post((Object)new BlobStoreStartedEvent(blobStore));
        }
        catch (BlobStoreException e) {
            this.startWithConfig(blobStore, currentBlobStoreConfiguration);
            throw e;
        }
        catch (Exception e) {
            this.log.error("Failed to update configuration", (Throwable)e);
            this.startWithConfig(blobStore, currentBlobStoreConfiguration);
            throw new BlobStoreException("Failed to start blob store with new configuration.", null);
        }
    }

    private void startWithConfig(BlobStore blobStore, BlobStoreConfiguration config) throws Exception {
        if (blobStore.isStarted()) {
            blobStore.stop();
            this.eventManager.post((Object)new BlobStoreStoppedEvent(blobStore));
        }
        blobStore.init(config);
        blobStore.start();
        this.eventManager.post((Object)new BlobStoreStartedEvent(blobStore));
    }

    @Guarded(by={"STARTED"})
    @Nullable
    public BlobStore get(String name) {
        Preconditions.checkNotNull((Object)name);
        BlobStore blobStore = this.stores.get(name);
        if (blobStore == null) {
            BlobStoreConfiguration configuration = this.store.read(name);
            if (Objects.isNull(configuration)) {
                return null;
            }
            try {
                BlobStore validBS = this.getBlobStoreForCreate(configuration);
                this.doCreate(validBS, configuration);
                blobStore = this.stores.get(name);
            }
            catch (Exception e) {
                throw new BlobStoreException("Could not start blobstore: " + name + ", reason: " + e.getMessage(), (Throwable)e, null);
            }
        }
        return blobStore;
    }

    @Guarded(by={"STARTED"})
    public void delete(String name) throws Exception {
        Preconditions.checkNotNull((Object)name);
        if (this.hasConflictingTasks(name)) {
            throw new IllegalStateException("BlobStore " + name + " is in use by a Change Repository Blob Store task");
        }
        if (((RepositoryManager)this.repositoryManagerProvider.get()).isBlobstoreUsed(name)) {
            throw new BlobStoreException("BlobStore " + name + " is in use and cannot be deleted", null);
        }
        this.forceDelete(name);
    }

    public boolean hasConflictingTasks(String blobStoreName) {
        return this.blobStoreTaskService.isAnyTaskInUseForBlobStore(blobStoreName);
    }

    @Guarded(by={"STARTED"})
    public void forceDelete(String name) throws Exception {
        Preconditions.checkNotNull((Object)name);
        this.freezeService.checkWritable("Unable to delete a BlobStore while database is frozen.");
        this.log.debug("Deleting BlobStore: {}", (Object)name);
        BlobStore blobStore = this.doForceDelete(name);
        if (!EventHelper.isReplicating()) {
            this.store.delete(blobStore.getBlobStoreConfiguration());
        }
        this.eventManager.post((Object)new BlobStoreDeletedEvent(blobStore));
        this.eventManager.post((Object)new BlobStoreDistributedConfigurationEvent(name, EventType.DELETED));
    }

    private BlobStore doForceDelete(String blobStoreName) {
        BlobStore blobStore = this.blobStore(blobStoreName);
        try {
            blobStore.shutdown();
            blobStore.remove();
        }
        catch (Exception e) {
            this.log.error("Error cleaning up blobStore {} while attempting to delete", (Object)blobStoreName, (Object)e);
        }
        this.untrack(blobStoreName);
        return blobStore;
    }

    public boolean exists(String name) {
        return this.stores.keySet().stream().anyMatch(key -> key.equalsIgnoreCase(name));
    }

    @VisibleForTesting
    BlobStore blobStore(String name) {
        BlobStore blobStore = this.stores.get(name);
        Preconditions.checkState((blobStore != null ? 1 : 0) != 0, (String)"Missing BlobStore: %s", (Object)name);
        return blobStore;
    }

    @VisibleForTesting
    void track(String name, BlobStore blobStore) {
        this.log.debug("Tracking: {}", (Object)name);
        this.stores.put(name, blobStore);
    }

    private void untrack(String name) {
        this.log.debug("Untracking: {}", (Object)name);
        this.stores.remove(name);
    }

    @Subscribe
    public void on(BlobStoreDistributedConfigurationEvent event) throws Exception {
        if (!EventHelper.isReplicating()) {
            return;
        }
        BlobStoreConfiguration configuration = this.store.read(event.getBlobStoreName());
        String blobStoreName = event.getBlobStoreName();
        EventType eventType = event.getEventType();
        this.log.debug("Received {} event for the {} blob store", (Object)eventType, (Object)blobStoreName);
        switch (eventType) {
            case CREATED: {
                this.validateConfiguration(configuration, true);
                this.doCreate(this.getBlobStoreForCreate(configuration), configuration);
                break;
            }
            case UPDATED: {
                this.doUpdate(this.getBlobStoreForUpdate(configuration), configuration);
                break;
            }
            case DELETED: {
                BlobStore deletedBlobStore = this.doForceDelete(blobStoreName);
                this.eventManager.post((Object)new BlobStoreDeletedEvent(deletedBlobStore));
                break;
            }
            default: {
                this.log.error("Unknown event type {}", (Object)eventType);
            }
        }
    }

    @Subscribe
    public void on(BlobStoreConfigurationCreatedEvent event) {
        this.handleReplication(event, (EventConsumer<BlobStoreConfigurationEvent>)((EventConsumer)e -> {
            BlobStore blobStore = this.create(e.getConfiguration());
        }));
    }

    @Subscribe
    public void on(BlobStoreConfigurationDeletedEvent event) {
        this.handleReplication(event, (EventConsumer<BlobStoreConfigurationEvent>)((EventConsumer)e -> this.forceDelete(e.getName())));
    }

    @Subscribe
    public void on(BlobStoreConfigurationUpdatedEvent event) {
        this.handleReplication(event, (EventConsumer<BlobStoreConfigurationEvent>)((EventConsumer)e -> {
            BlobStore blobStore = this.update(e.getConfiguration());
        }));
    }

    private void handleReplication(BlobStoreConfigurationEvent event, EventConsumer<BlobStoreConfigurationEvent> consumer) {
        if (!event.isLocal()) {
            try {
                consumer.accept((Object)event);
            }
            catch (Exception e) {
                this.log.error("Failed to replicate: {}", (Object)event, (Object)e);
            }
        }
    }

    public long blobStoreUsageCount(String blobStoreName) {
        long count = 0L;
        for (BlobStore otherBlobStore : this.stores.values()) {
            BlobStoreConfiguration otherBlobStoreConfig = otherBlobStore.getBlobStoreConfiguration();
            BlobStoreDescriptor otherBlobStoreDescriptor = this.blobStoreDescriptors.get(otherBlobStoreConfig.getType());
            if (!otherBlobStoreDescriptor.configHasDependencyOn(otherBlobStoreConfig, blobStoreName)) continue;
            ++count;
        }
        return count;
    }

    public boolean isConvertable(String blobStoreName) {
        BlobStore blobStore = this.get(blobStoreName);
        return blobStore != null && blobStore.isGroupable() && blobStore.isWritable() && !this.store.findParent(blobStore.getBlobStoreConfiguration().getName()).isPresent() && !this.hasConflictingTasks(blobStoreName);
    }

    public Optional<String> getParent(String blobStoreName) {
        BlobStore blobStore = this.get(blobStoreName);
        return blobStore == null ? Optional.empty() : this.store.findParent(blobStoreName).map(BlobStoreConfiguration::getName);
    }

    public BlobStoreConfiguration newConfiguration() {
        return this.store.newConfiguration();
    }

    public void validateConfiguration(BlobStoreConfiguration configuration, boolean sanitize) {
        Preconditions.checkNotNull((Object)configuration, (Object)"configuration param is NULL");
        BlobStoreDescriptor blobStoreDescriptor = this.blobStoreDescriptors.get(configuration.getType());
        if (sanitize) {
            blobStoreDescriptor.sanitizeConfig(configuration);
        }
        blobStoreDescriptor.validateConfig(configuration);
    }

    public BlobSession<?> openSession(String storeName) {
        return (BlobSession)Optional.ofNullable(this.get(storeName)).orElseThrow(() -> new BlobStoreNotFoundException(storeName)).openSession();
    }

    public Blob moveBlob(BlobId blobId, BlobStore srcBlobStore, BlobStore destBlobStore) {
        Preconditions.checkNotNull((Object)srcBlobStore);
        Preconditions.checkNotNull((Object)destBlobStore);
        BlobAttributes srcBlobAttributes = srcBlobStore.getBlobAttributes(blobId);
        boolean isSrcDeleted = srcBlobAttributes.isDeleted();
        Map headers = srcBlobAttributes.getHeaders();
        InputStream srcInputStream = this.inputStreamOfBlob(srcBlobStore, blobId);
        Blob newBlob = destBlobStore.create(srcInputStream, headers, blobId);
        destBlobStore.setBlobAttributes(blobId, srcBlobAttributes);
        this.ensureDeletedStateTransferred(blobId, srcBlobStore, destBlobStore, isSrcDeleted);
        this.log.debug("Created blobId {} in blob store '{}'", (Object)blobId, (Object)destBlobStore.getBlobStoreConfiguration().getName());
        try {
            srcBlobStore.deleteHard(blobId);
            this.log.debug("Removed blobId {} from blob store '{}'", (Object)blobId, (Object)srcBlobStore.getBlobStoreConfiguration().getName());
        }
        catch (BlobStoreException e) {
            this.log.warn("Failed to remove blobId {} from blob store '{}'", new Object[]{blobId, srcBlobStore.getBlobStoreConfiguration().getName(), e});
        }
        return newBlob;
    }

    private void ensureDeletedStateTransferred(BlobId blobId, BlobStore srcBlobStore, BlobStore destBlobStore, boolean isSrcDeleted) {
        BlobAttributes currentSrcAttributes = srcBlobStore.getBlobAttributes(blobId);
        if (currentSrcAttributes != null && currentSrcAttributes.isDeleted() != isSrcDeleted) {
            BlobAttributes outOfDateAttributes = destBlobStore.getBlobAttributes(blobId);
            outOfDateAttributes.setDeleted(!isSrcDeleted);
            destBlobStore.setBlobAttributes(blobId, outOfDateAttributes);
        }
    }

    private InputStream inputStreamOfBlob(BlobStore blobStore, BlobId blobId) {
        try {
            return Optional.of(blobId).map(r -> blobStore.get(r, true)).map(Blob::getInputStream).orElseThrow(() -> new IllegalStateException(String.format("Unable to get input stream from source %S with blobId: %s", blobStore.getBlobStoreConfiguration().getName(), blobId)));
        }
        catch (BlobStoreException ex) {
            throw new BlobInputStreamException((Throwable)ex, ex.getBlobId());
        }
    }

    public Map<String, BlobStore> getByName() {
        return Collections.unmodifiableMap(this.stores);
    }
}

