/*
 * Decompiled with CFR 0.152.
 */
package org.sonatype.nexus.blobstore;

import com.codahale.metrics.MetricRegistry;
import com.codahale.metrics.Timer;
import com.google.common.base.Preconditions;
import java.io.IOException;
import java.io.InputStream;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;
import java.util.regex.Pattern;
import javax.annotation.Nullable;
import javax.inject.Inject;
import org.sonatype.nexus.blobstore.AttributesLocation;
import org.sonatype.nexus.blobstore.BlobIdLocationResolver;
import org.sonatype.nexus.blobstore.MemoryBlobSession;
import org.sonatype.nexus.blobstore.PerformanceLogger;
import org.sonatype.nexus.blobstore.api.Blob;
import org.sonatype.nexus.blobstore.api.BlobAttributes;
import org.sonatype.nexus.blobstore.api.BlobId;
import org.sonatype.nexus.blobstore.api.BlobSession;
import org.sonatype.nexus.blobstore.api.BlobStore;
import org.sonatype.nexus.blobstore.api.BlobStoreConfiguration;
import org.sonatype.nexus.blobstore.api.BlobStoreUsageChecker;
import org.sonatype.nexus.common.log.DryRunPrefix;
import org.sonatype.nexus.common.stateguard.Guarded;
import org.sonatype.nexus.common.stateguard.StateGuardLifecycleSupport;
import org.sonatype.nexus.common.stateguard.Transitions;

public abstract class BlobStoreSupport<T extends AttributesLocation>
extends StateGuardLifecycleSupport
implements BlobStore {
    public static final String CONTENT_TMP_PATH = "/content/tmp/";
    private final Map<String, Timer> timers = new ConcurrentHashMap<String, Timer>();
    protected final PerformanceLogger performanceLogger = new PerformanceLogger();
    private MetricRegistry metricRegistry;
    protected final BlobIdLocationResolver blobIdLocationResolver;
    protected final DryRunPrefix dryRunPrefix;
    protected BlobStoreConfiguration blobStoreConfiguration;
    protected static final Pattern UUID_PATTERN = Pattern.compile(".*vol-\\d{2}[/\\\\]chap-\\d{2}[/\\\\]\\b[0-9a-f]{8}\\b-[0-9a-f]{4}-[0-9a-f]{4}-[0-9a-f]{4}-\\b[0-9a-f]{12}\\b.properties$", 2);
    public static final int MAX_NAME_LENGTH = 255;
    public static final int MIN_NAME_LENGTH = 1;

    public BlobStoreSupport(BlobIdLocationResolver blobIdLocationResolver, DryRunPrefix dryRunPrefix) {
        this.blobIdLocationResolver = (BlobIdLocationResolver)Preconditions.checkNotNull((Object)blobIdLocationResolver);
        this.dryRunPrefix = (DryRunPrefix)Preconditions.checkNotNull((Object)dryRunPrefix);
    }

    @Inject
    public void setMetricRegistry(MetricRegistry metricRegistry) {
        this.metricRegistry = metricRegistry;
    }

    protected BlobId getBlobId(Map<String, String> headers, @Nullable BlobId blobId) {
        return Optional.ofNullable(blobId).orElseGet(() -> this.blobIdLocationResolver.fromHeaders(headers));
    }

    private void checkIsWritable() {
        Preconditions.checkState((boolean)this.isWritable(), (Object)"Operation not permitted when blob store is not writable");
    }

    @Guarded(by={"STARTED"})
    public BlobSession<?> openSession() {
        return new MemoryBlobSession(this);
    }

    @Guarded(by={"STARTED"})
    public Blob create(InputStream blobData, Map<String, String> headers) {
        return this.create(blobData, headers, null);
    }

    @Guarded(by={"STARTED"})
    public Blob create(InputStream blobData, Map<String, String> headers, @Nullable BlobId blobId) {
        Preconditions.checkNotNull((Object)blobData);
        Preconditions.checkNotNull(headers);
        this.checkIsWritable();
        Preconditions.checkArgument((boolean)headers.containsKey("BlobStore.blob-name"), (String)"Missing header: %s", (Object)"BlobStore.blob-name");
        Preconditions.checkArgument((boolean)headers.containsKey("BlobStore.created-by"), (String)"Missing header: %s", (Object)"BlobStore.created-by");
        long start = System.nanoTime();
        Blob blob = null;
        try {
            blob = this.doCreate(blobData, headers, blobId);
        }
        finally {
            long elapsed = System.nanoTime() - start;
            this.updateTimer("create", elapsed);
            if (blob != null) {
                this.performanceLogger.logCreate(blob, elapsed);
            }
        }
        return blob;
    }

    protected abstract Blob doCreate(InputStream var1, Map<String, String> var2, @Nullable BlobId var3);

    @Guarded(by={"STARTED"})
    public boolean delete(BlobId blobId, String reason) {
        Preconditions.checkNotNull((Object)blobId);
        long start = System.nanoTime();
        try {
            boolean bl = this.doDelete(blobId, reason);
            return bl;
        }
        finally {
            long elapsed = System.nanoTime() - start;
            this.updateTimer("delete", elapsed);
            this.performanceLogger.logDelete(elapsed);
        }
    }

    protected abstract boolean doDelete(BlobId var1, String var2);

    @Guarded(by={"STARTED"})
    public boolean undelete(@Nullable BlobStoreUsageChecker inUseChecker, BlobId blobId, BlobAttributes attributes, boolean isDryRun) {
        Preconditions.checkNotNull((Object)attributes);
        String logPrefix = isDryRun ? this.dryRunPrefix.get() : "";
        Optional<String> blobName = Optional.of(attributes).map(BlobAttributes::getProperties).map(p -> p.getProperty("@BlobStore.blob-name"));
        if (!blobName.isPresent()) {
            this.log.error("Property not present: {}, for blob id: {}, at path: {}", new Object[]{"@BlobStore.blob-name", blobId, this.attributePathString(blobId)});
            return false;
        }
        if (attributes.isDeleted() && inUseChecker != null && inUseChecker.test((BlobStore)this, blobId, blobName.get())) {
            String deletedReason = attributes.getDeletedReason();
            if (!isDryRun) {
                attributes.setDeleted(false);
                attributes.setDeletedReason(null);
                try {
                    this.doUndelete(blobId, attributes);
                    attributes.store();
                }
                catch (IOException e) {
                    this.log.error("Error while un-deleting blob id: {}, deleted reason: {}, blob store: {}, blob name: {}", new Object[]{blobId, deletedReason, this.blobStoreConfiguration.getName(), blobName.get(), e});
                }
            }
            this.log.warn("{}Soft-deleted blob still in use, un-deleting blob id: {}, deleted reason: {}, blob store: {}, blob name: {}", new Object[]{logPrefix, blobId, deletedReason, this.blobStoreConfiguration.getName(), blobName.get()});
            return true;
        }
        return false;
    }

    protected abstract String attributePathString(BlobId var1);

    protected void doUndelete(BlobId blobId, BlobAttributes attributes) {
    }

    @Guarded(by={"STARTED"})
    public boolean deleteHard(BlobId blobId) {
        Preconditions.checkNotNull((Object)blobId);
        long start = System.nanoTime();
        try {
            boolean bl = this.doDeleteHard(blobId);
            return bl;
        }
        finally {
            this.updateTimer("deleteHard", System.nanoTime() - start);
        }
    }

    protected abstract boolean doDeleteHard(BlobId var1);

    @Guarded(by={"STARTED"})
    public synchronized void compact(@Nullable BlobStoreUsageChecker inUseChecker) {
        long start = System.nanoTime();
        try {
            this.doCompact(inUseChecker);
        }
        finally {
            this.updateTimer("compact", System.nanoTime() - start);
        }
    }

    protected void doCompact(@Nullable BlobStoreUsageChecker inUseChecker) {
    }

    @Guarded(by={"STARTED"})
    public synchronized void deleteTempFiles(Integer daysOlderThan) {
        long start = System.nanoTime();
        try {
            this.doDeleteTempFiles(daysOlderThan);
        }
        finally {
            this.updateTimer("delete temp files", System.nanoTime() - start);
        }
    }

    protected void doDeleteTempFiles(Integer daysOlderThan) {
    }

    public void init(BlobStoreConfiguration configuration) {
        this.blobStoreConfiguration = configuration;
        this.performanceLogger.setBlobStoreName(configuration.getName());
        this.doInit(this.blobStoreConfiguration);
    }

    protected abstract void doInit(BlobStoreConfiguration var1);

    public BlobStoreConfiguration getBlobStoreConfiguration() {
        return this.blobStoreConfiguration;
    }

    protected abstract BlobAttributes getBlobAttributes(T var1) throws IOException;

    protected String getBlobIdFromAttributeFilePath(T attributeFilePath) {
        if (UUID_PATTERN.matcher(attributeFilePath.getFullPath()).matches()) {
            String filename = attributeFilePath.getFileName();
            return filename.substring(0, filename.length() - ".properties".length());
        }
        try {
            BlobAttributes fileBlobAttributes = this.getBlobAttributes(attributeFilePath);
            return this.blobIdLocationResolver.fromHeaders(fileBlobAttributes.getHeaders()).asUniqueString();
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    private void updateTimer(String name, long value) {
        if (this.metricRegistry != null) {
            Timer timer = this.timers.computeIfAbsent(name, key -> this.metricRegistry.timer(String.valueOf(((Object)((Object)this)).getClass().getName().replaceAll("\\$.*", "")) + '.' + name + ".timer"));
            timer.update(value, TimeUnit.NANOSECONDS);
        }
    }

    public boolean isEmpty() {
        return !this.getBlobIdStream().findAny().isPresent();
    }

    @Transitions(to="SHUTDOWN")
    public void shutdown() throws Exception {
        if (this.isStarted()) {
            this.doStop();
        }
    }
}

