/*
 * Decompiled with CFR 0.152.
 */
package org.sonatype.nexus.blobstore;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import java.io.IOException;
import java.util.EnumMap;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import java.util.stream.Stream;
import org.sonatype.nexus.blobstore.AccumulatingBlobStoreMetrics;
import org.sonatype.nexus.blobstore.BlobStoreMetricsNotAvailableException;
import org.sonatype.nexus.blobstore.UnavailableBlobStoreMetrics;
import org.sonatype.nexus.blobstore.api.BlobStore;
import org.sonatype.nexus.blobstore.api.BlobStoreMetrics;
import org.sonatype.nexus.blobstore.api.OperationMetrics;
import org.sonatype.nexus.blobstore.api.OperationType;
import org.sonatype.nexus.blobstore.api.metrics.BlobStoreMetricsService;
import org.sonatype.nexus.common.node.NodeAccess;
import org.sonatype.nexus.common.property.ImplicitSourcePropertiesFile;
import org.sonatype.nexus.common.stateguard.Guarded;
import org.sonatype.nexus.common.stateguard.StateGuardLifecycleSupport;
import org.sonatype.nexus.scheduling.PeriodicJobService;

public abstract class BlobStoreMetricsStoreSupport<T extends ImplicitSourcePropertiesFile>
extends StateGuardLifecycleSupport
implements BlobStoreMetricsService {
    private static final int METRICS_LOADING_DELAY_MILLIS = 200;
    public static final int MAXIMUM_TRIES = 3;
    @VisibleForTesting
    public static final String METRICS_FILENAME = "metrics.properties";
    @VisibleForTesting
    public static final String TOTAL_SIZE_PROP_NAME = "totalSize";
    @VisibleForTesting
    public static final String BLOB_COUNT_PROP_NAME = "blobCount";
    private static final String TOTAL_SUCCESSFUL_REQUESTS_PROP_NAME = "_totalSuccessfulRequests";
    private static final String TOTAL_ERROR_REQUESTS_PROP_NAME = "_totalErrorRequests";
    private static final String TOTAL_TIME_ON_REQUEST_PROP_NAME = "_totalTimeOnRequests";
    public static final String TOTAL_SIZE_ON_REQUEST_PROP_NAME = "_totalSizeOnRequests";
    private static final int METRICS_FLUSH_PERIOD_SECONDS = 2;
    protected AtomicLong blobCount;
    protected AtomicLong totalSize;
    protected AtomicBoolean dirty;
    protected BlobStore blobStore;
    protected final NodeAccess nodeAccess;
    protected T properties;
    protected final PeriodicJobService jobService;
    protected PeriodicJobService.PeriodicJob metricsWritingJob;
    private Map<OperationType, OperationMetrics> operationMetrics;

    public BlobStoreMetricsStoreSupport(NodeAccess nodeAccess, PeriodicJobService jobService) {
        this.nodeAccess = (NodeAccess)Preconditions.checkNotNull((Object)nodeAccess);
        this.jobService = (PeriodicJobService)Preconditions.checkNotNull((Object)jobService);
    }

    protected void doStart() throws Exception {
        this.blobCount = new AtomicLong();
        this.operationMetrics = new EnumMap<OperationType, OperationMetrics>(OperationType.class);
        OperationType[] operationTypeArray = OperationType.values();
        int n = operationTypeArray.length;
        int n2 = 0;
        while (n2 < n) {
            OperationType type = operationTypeArray[n2];
            this.operationMetrics.put(type, new OperationMetrics());
            ++n2;
        }
        this.totalSize = new AtomicLong();
        this.dirty = new AtomicBoolean();
        this.properties = this.getProperties();
        if (this.properties.exists()) {
            this.log.info("Loading blob store metrics file {}", this.properties);
            this.properties.load();
            this.readProperties();
        } else {
            this.log.info("Blob store metrics file {} not found - initializing at zero.", this.properties);
            this.flushProperties();
        }
        this.jobService.startUsing();
        this.metricsWritingJob = this.jobService.schedule(() -> {
            try {
                if (this.dirty.compareAndSet(true, false)) {
                    this.flushProperties();
                }
            }
            catch (Exception e) {
                this.log.error("Cannot write blob store metrics", (Throwable)e);
            }
        }, 2);
    }

    protected void doStop() throws Exception {
        this.blobStore = null;
        this.metricsWritingJob.cancel();
        this.metricsWritingJob = null;
        this.jobService.stopUsing();
        this.blobCount = null;
        this.operationMetrics.clear();
        this.totalSize = null;
        this.dirty = null;
        this.properties = null;
    }

    protected abstract T getProperties();

    protected abstract AccumulatingBlobStoreMetrics getAccumulatingBlobStoreMetrics() throws BlobStoreMetricsNotAvailableException;

    protected abstract Stream<T> backingFiles() throws BlobStoreMetricsNotAvailableException;

    @VisibleForTesting
    public synchronized void flushProperties() throws IOException {
        this.updateProperties();
        this.log.trace("Writing blob store metrics to {}", this.properties);
        this.properties.store();
    }

    protected BlobStoreMetrics getCombinedMetrics(Stream<T> blobStoreMetricsFiles) throws BlobStoreMetricsNotAvailableException {
        AccumulatingBlobStoreMetrics blobStoreMetrics = this.getAccumulatingBlobStoreMetrics();
        blobStoreMetricsFiles.forEach(metricsFile -> {
            Stream.iterate(1, i -> i + 1).limit(3L).forEach(currentTry -> {
                try {
                    metricsFile.load();
                }
                catch (IOException e) {
                    this.log.debug("Unable to load properties file {}. Try number {} of {}.", new Object[]{metricsFile, currentTry, 3, e});
                    if (currentTry >= 3) {
                        throw new RuntimeException("Failed to load blob store metrics from " + metricsFile, e);
                    }
                    try {
                        TimeUnit.MILLISECONDS.sleep(200L);
                    }
                    catch (InterruptedException e1) {
                        throw new RuntimeException(e1);
                    }
                }
            });
            blobStoreMetrics.addBlobCount(Long.parseLong(metricsFile.getProperty(BLOB_COUNT_PROP_NAME, "0")));
            blobStoreMetrics.addTotalSize(Long.parseLong(metricsFile.getProperty(TOTAL_SIZE_PROP_NAME, "0")));
        });
        return blobStoreMetrics;
    }

    public void setBlobStore(BlobStore blobStore) {
        Preconditions.checkState((this.blobStore == null ? 1 : 0) != 0, (Object)"Do not initialize twice");
        Preconditions.checkNotNull((Object)blobStore);
        this.blobStore = blobStore;
    }

    @Guarded(by={"STARTED"})
    public BlobStoreMetrics getMetrics() {
        try {
            return this.getCombinedMetrics(this.backingFiles());
        }
        catch (BlobStoreMetricsNotAvailableException e) {
            this.log.error("Blob store metrics cannot be accessed", (Throwable)e);
            return UnavailableBlobStoreMetrics.getInstance();
        }
    }

    public Map<OperationType, OperationMetrics> getOperationMetrics() {
        return this.operationMetrics;
    }

    @Guarded(by={"STARTED"})
    public void recordAddition(long size) {
        this.blobCount.incrementAndGet();
        this.totalSize.addAndGet(size);
        this.dirty.set(true);
    }

    @Guarded(by={"STARTED"})
    public void recordDeletion(long size) {
        this.blobCount.decrementAndGet();
        this.totalSize.addAndGet(-size);
        this.dirty.set(true);
    }

    private void updateProperties() {
        this.properties.setProperty(TOTAL_SIZE_PROP_NAME, this.totalSize.toString());
        this.properties.setProperty(BLOB_COUNT_PROP_NAME, this.blobCount.toString());
        for (Map.Entry<OperationType, OperationMetrics> operationMetricByType : this.operationMetrics.entrySet()) {
            OperationType type = operationMetricByType.getKey();
            OperationMetrics operationMetric = operationMetricByType.getValue();
            this.properties.setProperty(type + TOTAL_SUCCESSFUL_REQUESTS_PROP_NAME, String.valueOf(operationMetric.getSuccessfulRequests()));
            this.properties.setProperty(type + TOTAL_ERROR_REQUESTS_PROP_NAME, String.valueOf(operationMetric.getErrorRequests()));
            this.properties.setProperty(type + TOTAL_TIME_ON_REQUEST_PROP_NAME, String.valueOf(operationMetric.getTimeOnRequests()));
            this.properties.setProperty(type + TOTAL_SIZE_ON_REQUEST_PROP_NAME, String.valueOf(operationMetric.getBlobSize()));
        }
    }

    private void readProperties() {
        String count;
        String size = this.properties.getProperty(TOTAL_SIZE_PROP_NAME);
        if (size != null) {
            this.totalSize.set(Long.parseLong(size));
        }
        if ((count = this.properties.getProperty(BLOB_COUNT_PROP_NAME)) != null) {
            this.blobCount.set(Long.parseLong(count));
        }
        for (Map.Entry<OperationType, OperationMetrics> operationMetricByType : this.operationMetrics.entrySet()) {
            String totalSizeOnRequests;
            String totalTimeOnRequests;
            String totalErrorRequests;
            OperationType type = operationMetricByType.getKey();
            OperationMetrics operationMetric = operationMetricByType.getValue();
            String totalSuccessfulRequests = this.properties.getProperty(type + TOTAL_SUCCESSFUL_REQUESTS_PROP_NAME);
            if (totalSuccessfulRequests != null) {
                operationMetric.setSuccessfulRequests(Long.parseLong(totalSuccessfulRequests));
            }
            if ((totalErrorRequests = this.properties.getProperty(type + TOTAL_ERROR_REQUESTS_PROP_NAME)) != null) {
                operationMetric.setErrorRequests(Long.parseLong(totalErrorRequests));
            }
            if ((totalTimeOnRequests = this.properties.getProperty(type + TOTAL_TIME_ON_REQUEST_PROP_NAME)) != null) {
                operationMetric.setTimeOnRequests(Long.parseLong(totalTimeOnRequests));
            }
            if ((totalSizeOnRequests = this.properties.getProperty(type + TOTAL_SIZE_ON_REQUEST_PROP_NAME)) == null) continue;
            operationMetric.setBlobSize(Long.parseLong(totalSizeOnRequests));
        }
    }

    public void clearCountMetrics() {
        this.totalSize.set(0L);
        this.blobCount.set(0L);
    }

    public void clearOperationMetrics() {
        this.getOperationMetricsDelta().values().forEach(OperationMetrics::clear);
        this.getOperationMetrics().values().forEach(OperationMetrics::clear);
    }

    public abstract void remove();
}

