/*
 * Decompiled with CFR 0.152.
 */
package org.sonatype.nexus.repository.rubygems.orient.internal.hosted;

import com.google.common.base.Preconditions;
import com.google.common.base.Throwables;
import com.google.common.collect.ImmutableMap;
import com.google.common.hash.HashCode;
import com.google.common.reflect.TypeToken;
import java.io.IOException;
import java.io.InputStream;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.Collections;
import java.util.Map;
import java.util.Optional;
import javax.annotation.Nullable;
import javax.inject.Inject;
import javax.inject.Named;
import org.sonatype.nexus.blobstore.api.Blob;
import org.sonatype.nexus.blobstore.api.BlobStore;
import org.sonatype.nexus.blobstore.api.BlobStoreManager;
import org.sonatype.nexus.common.entity.Entity;
import org.sonatype.nexus.common.hash.HashAlgorithm;
import org.sonatype.nexus.logging.task.ProgressLogIntervalHelper;
import org.sonatype.nexus.repository.Repository;
import org.sonatype.nexus.repository.RepositoryTaskSupport;
import org.sonatype.nexus.repository.rubygems.internal.hosted.RubygemsGenerateSha256ChecksumsTask;
import org.sonatype.nexus.repository.storage.Asset;
import org.sonatype.nexus.repository.storage.AssetEntityAdapter;
import org.sonatype.nexus.repository.storage.StorageFacet;
import org.sonatype.nexus.repository.storage.StorageTx;
import org.sonatype.nexus.repository.transaction.TransactionalStoreMetadata;
import org.sonatype.nexus.scheduling.CancelableHelper;
import org.sonatype.nexus.transaction.UnitOfWork;

@Named
public class OrientRubygemsGenerateSha256ChecksumsTask
extends RepositoryTaskSupport
implements RubygemsGenerateSha256ChecksumsTask {
    private static final String BEGINNING_ID = "#-1:-1";
    private static final String ASSETS_WHERE = "@rid > :rid";
    private static final String ASSETS_SUFFIX = "ORDER BY @rid LIMIT :limit";
    private static final int BATCH_SIZE = 100;
    private final AssetEntityAdapter assetEntityAdapter;
    private int bufferSize;
    private BlobStoreManager blobStoreManager;
    private MessageDigest messageDigest;

    public String getMessage() {
        return "Generating SHA256 hashes";
    }

    @Inject
    public OrientRubygemsGenerateSha256ChecksumsTask(AssetEntityAdapter assetEntityAdapter) {
        this.assetEntityAdapter = (AssetEntityAdapter)Preconditions.checkNotNull((Object)assetEntityAdapter);
    }

    @Inject
    public void init(@Named(value="${nexus.calculateChecksums.bufferSize:-32768}") @Named(value="${nexus.calculateChecksums.bufferSize:-32768}") int bufferSize, BlobStoreManager blobStoreManager) throws NoSuchAlgorithmException {
        this.bufferSize = Math.max(4096, bufferSize);
        this.blobStoreManager = (BlobStoreManager)Preconditions.checkNotNull((Object)blobStoreManager);
        this.messageDigest = MessageDigest.getInstance(HashAlgorithm.SHA256.name());
    }

    private BlobStore getBlobStore(Repository repository) {
        String name = (String)repository.getConfiguration().attributes("storage").get("blobStoreName", String.class);
        return this.blobStoreManager.get(name);
    }

    protected void execute(Repository repository) {
        this.log.info("Checking for missing SHA256 checksums in repository: {}", (Object)repository.getName());
        BlobStore blobStore = this.getBlobStore(repository);
        ResultCount resultCount = new ResultCount();
        try {
            Throwable throwable = null;
            Object var5_7 = null;
            try (ProgressLogIntervalHelper progressLogger = new ProgressLogIntervalHelper(this.log, 60);){
                String lastId = BEGINNING_ID;
                while (lastId != null) {
                    try {
                        lastId = this.processBatch(repository, blobStore, lastId, resultCount, progressLogger);
                    }
                    catch (Exception e) {
                        Throwables.propagateIfPossible((Throwable)e, RuntimeException.class);
                        throw new RuntimeException(e);
                    }
                }
                this.log.info("Completed processing a total of {} assets. {} updated, {} skipped, {} errors.", new Object[]{resultCount.total, resultCount.updated, resultCount.skipped, resultCount.error});
            }
            catch (Throwable throwable2) {
                if (throwable == null) {
                    throwable = throwable2;
                } else if (throwable != throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
        }
        catch (Exception e) {
            Throwables.propagateIfPossible((Throwable)e, RuntimeException.class);
            throw new RuntimeException(e);
        }
    }

    @Nullable
    private String processBatch(Repository repository, BlobStore blobStore, String lastId, ResultCount resultCount, ProgressLogIntervalHelper progressLogger) throws Exception {
        this.log.debug("Processing next batch of rubygems assets for update. Starting at id = {} with max batch size = {}", (Object)lastId, (Object)100);
        return (String)TransactionalStoreMetadata.operation.withDb(((StorageFacet)repository.facet(StorageFacet.class)).txSupplier()).throwing(Exception.class).call(() -> {
            Iterable<Asset> assets = this.readAssets(repository, lastId);
            return this.updateAssets(repository, blobStore, assets, resultCount, progressLogger);
        });
    }

    private String updateAssets(Repository repository, BlobStore blobStore, Iterable<Asset> assets, ResultCount resultCount, ProgressLogIntervalHelper progressLogger) {
        String lastId = null;
        for (Asset asset : assets) {
            lastId = this.assetEntityAdapter.recordIdentity((Entity)asset).toString();
            this.processAssetChecksums(blobStore, asset, resultCount, progressLogger);
        }
        return lastId;
    }

    private Iterable<Asset> readAssets(Repository repository, String lastId) {
        StorageTx storageTx = (StorageTx)UnitOfWork.currentTx();
        ImmutableMap parameters = ImmutableMap.of((Object)"rid", (Object)lastId, (Object)"limit", (Object)100);
        return storageTx.findAssets(ASSETS_WHERE, (Map)parameters, Collections.singletonList(repository), ASSETS_SUFFIX);
    }

    private void processAssetChecksums(BlobStore blobStore, Asset asset, ResultCount resultCount, ProgressLogIntervalHelper progressLogger) {
        CancelableHelper.checkCancellation();
        String assetPath = asset.name();
        this.log.debug("Checking asset: {}", (Object)assetPath);
        if (asset.getChecksum(HashAlgorithm.SHA256) != null) {
            this.log.debug("SHA256 checksum already present for asset: {}. Skipping", (Object)assetPath);
            ++resultCount.skipped;
            return;
        }
        StorageTx tx = (StorageTx)UnitOfWork.currentTx();
        Blob blob = Optional.ofNullable(asset.blobRef()).map(arg_0 -> ((StorageTx)tx).getBlob(arg_0)).orElse(null);
        if (blob == null) {
            this.log.debug("No Blob associated with asset: {}. Skipping", (Object)assetPath);
            ++resultCount.skipped;
            return;
        }
        Map checksums = (Map)asset.attributes().require("checksum", (TypeToken)new TypeToken<Map<String, String>>(){});
        String sha256 = this.calculateBlobChecksum(blob, assetPath);
        if (sha256 != null) {
            checksums.put(HashAlgorithm.SHA256.name(), sha256);
            this.log.debug("Updating blob SHA256 checksum for asset: {}, blobID: {}, checksum: {}", new Object[]{assetPath, blob.getId(), sha256});
            asset.attributes().set("checksum", (Object)checksums);
            tx.saveAsset(asset);
            ++resultCount.updated;
        } else {
            this.log.debug("Unable to create SHA256 checksum for asset: {}, blobID: {}", (Object)assetPath, (Object)blob.getId());
            ++resultCount.error;
        }
        ++resultCount.total;
        progressLogger.info("Elapsed time: {}. Processed {} assets. {} updated, {} skipped, {} errors.", new Object[]{progressLogger.getElapsed(), resultCount.total, resultCount.updated, resultCount.skipped, resultCount.error});
    }

    private String calculateBlobChecksum(Blob blob, String assetPath) {
        this.log.debug("Calculating SHA256 checksum for {}", (Object)assetPath);
        try {
            Throwable throwable = null;
            Object var4_6 = null;
            try (InputStream inputStream = blob.getInputStream();){
                int bytesRead = 0;
                byte[] buffer = new byte[this.bufferSize];
                while (bytesRead != -1) {
                    bytesRead = inputStream.read(buffer);
                    if (bytesRead <= 0) continue;
                    this.messageDigest.update(buffer, 0, bytesRead);
                }
                return HashCode.fromBytes((byte[])this.messageDigest.digest()).toString();
            }
            catch (Throwable throwable2) {
                if (throwable == null) {
                    throwable = throwable2;
                } else if (throwable != throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
        }
        catch (IOException e) {
            this.log.warn(String.format("Exception whilst calculating SHA256 checksum for %s: %s", assetPath, e.getLocalizedMessage()), (Throwable)(this.log.isDebugEnabled() ? e : null));
            return null;
        }
    }

    protected boolean appliesTo(Repository repository) {
        return "rubygems".equals(repository.getFormat().getValue()) && !this.isGroupRepository(repository);
    }

    private static class ResultCount {
        int updated;
        int skipped;
        int error;
        int total;

        private ResultCount() {
        }
    }
}

