/*
 * Decompiled with CFR 0.152.
 */
package com.sonatype.analytics.internal.contentusage;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.sonatype.analytics.internal.AnalyticsConstants;
import com.sonatype.analytics.internal.evaluator.ContentUsageEvaluatorManager;
import com.sonatype.analytics.internal.metrics.AggregatedMetric;
import com.sonatype.analytics.internal.metrics.AggregatedMetricsStore;
import com.sonatype.analytics.internal.metrics.HistoricalLoginInfoStore;
import com.sonatype.analytics.internal.metrics.MetricStore;
import com.sonatype.analytics.internal.service.CommonQueryDataService;
import java.time.OffsetDateTime;
import java.time.ZoneOffset;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;
import javax.inject.Inject;
import javax.inject.Named;
import org.sonatype.nexus.common.event.EventManager;
import org.sonatype.nexus.common.event.UsageMetricsAggregatedEvent;
import org.sonatype.nexus.logging.task.TaskLogType;
import org.sonatype.nexus.logging.task.TaskLogging;
import org.sonatype.nexus.repository.manager.RepositoryManager;
import org.sonatype.nexus.scheduling.Cancelable;
import org.sonatype.nexus.scheduling.CancelableHelper;
import org.sonatype.nexus.scheduling.TaskSupport;

@Named
@TaskLogging(value=TaskLogType.NEXUS_LOG_ONLY)
public class ContentUsageAggregationTask
extends TaskSupport
implements Cancelable {
    static String TASK_ID = "content.usage.aggregation";
    static String TASK_NAME = "Content usage aggregation";
    static String SCHEDULE = "0 1 * * * ? *";
    private final OffsetDateTime aggregationTime = this.currentTime().withOffsetSameInstant(ZoneOffset.UTC).withMinute(0).withSecond(0).withNano(0);
    private final AggregatedMetricsStore aggregatedMetricStore;
    private final HistoricalLoginInfoStore loginStore;
    private final MetricStore metricStore;
    private final RepositoryManager repositoryManager;
    private final CommonQueryDataService commonQueryDataService;
    private Map<String, List<AggregatedMetric>> recentMetricsByName;
    private final ContentUsageEvaluatorManager contentUsageEvaluatorManager;
    private final EventManager eventManager;

    @Inject
    public ContentUsageAggregationTask(HistoricalLoginInfoStore loginStore, MetricStore metricStore, AggregatedMetricsStore aggregatedMetricStore, RepositoryManager repositoryManager, CommonQueryDataService commonQueryDataService, ContentUsageEvaluatorManager contentUsageEvaluatorManager, EventManager eventManager) {
        this.aggregatedMetricStore = (AggregatedMetricsStore)Preconditions.checkNotNull((Object)aggregatedMetricStore);
        this.loginStore = (HistoricalLoginInfoStore)Preconditions.checkNotNull((Object)loginStore);
        this.metricStore = (MetricStore)Preconditions.checkNotNull((Object)metricStore);
        this.repositoryManager = Objects.requireNonNull(repositoryManager);
        this.commonQueryDataService = Objects.requireNonNull(commonQueryDataService);
        this.contentUsageEvaluatorManager = Objects.requireNonNull(contentUsageEvaluatorManager);
        this.eventManager = Objects.requireNonNull(eventManager);
    }

    public String getMessage() {
        return "Aggregating metric data";
    }

    protected Object execute() {
        this.log.debug("Aggregating metrics as of {}", (Object)this.aggregationTime);
        this.recentMetricsByName = this.aggregatedMetricStore.getRecordsSince(this.aggregationTime.minusDays(1L)).stream().collect(Collectors.groupingBy(AggregatedMetric::getMetricName));
        int i = 0;
        while (i < 24) {
            this.processMetrics(this.aggregationTime.minusHours(i), i == 0);
            ++i;
        }
        this.pruneLogins();
        this.pruneMetricData("content_request_count");
        this.pruneMetricData("component_total_count");
        this.contentUsageEvaluatorManager.evaluateUsage();
        this.eventManager.post((Object)new UsageMetricsAggregatedEvent());
        return null;
    }

    private void processMetrics(OffsetDateTime end, boolean currentTimestamp) {
        this.sumMetric("content_request_count", end.minusHours(1L), end, AnalyticsConstants.AggregationPeriod.hour);
        this.sumMetric("content_request_count", end.minusHours(24L), end, AnalyticsConstants.AggregationPeriod.last_24h);
        this.maxMetric("component_total_count", end.minusHours(1L), end, AnalyticsConstants.AggregationPeriod.hour, currentTimestamp);
        this.processUniqueUserLogins(end.minusHours(1L), end, AnalyticsConstants.AggregationPeriod.hour);
        Long uniqueLoginsDay = this.processUniqueUserLogins(end.minusDays(1L), end, AnalyticsConstants.AggregationPeriod.last_24h);
        this.processUniqueUserLogins(end.minusDays(30L), end, AnalyticsConstants.AggregationPeriod.last_30d);
        if (ContentUsageAggregationTask.isDayStart(end)) {
            if (!this.isMetricAlreadyRecorded("content_request_count", end, AnalyticsConstants.AggregationPeriod.day)) {
                this.sumMetric("content_request_count", end.minusDays(1L), end, AnalyticsConstants.AggregationPeriod.day);
            }
            if (!this.isMetricAlreadyRecorded("component_total_count", end, AnalyticsConstants.AggregationPeriod.day)) {
                this.maxMetric("component_total_count", end.minusDays(1L), end, AnalyticsConstants.AggregationPeriod.day, currentTimestamp);
            }
            if (!this.isMetricAlreadyRecorded("unique_user_count", end, AnalyticsConstants.AggregationPeriod.day)) {
                this.aggregatedMetricStore.addMetricData("unique_user_count", uniqueLoginsDay, end, AnalyticsConstants.AggregationPeriod.day.toString());
            }
        }
    }

    private void maxMetric(String metricName, OffsetDateTime start, OffsetDateTime end, AnalyticsConstants.AggregationPeriod periodName, boolean currentTimestamp) {
        Long max;
        CancelableHelper.checkCancellation();
        if (this.isMetricAlreadyRecorded(metricName, end, periodName)) {
            this.log.debug("Found persisted {} @ {} period {}", new Object[]{metricName, end, periodName});
            return;
        }
        if (currentTimestamp && "component_total_count".equals(metricName)) {
            this.log.debug("Finding total {} for {} to {}", new Object[]{metricName, start, end});
            max = this.getTotalComponents();
        } else {
            this.log.debug("Finding max {} for {} to {}", new Object[]{metricName, start, end});
            max = this.metricStore.getMaxMetricValue(metricName, start, end);
        }
        this.log.debug("Found {} {}", (Object)max, (Object)metricName);
        this.aggregatedMetricStore.addMetricData(metricName, max, end, periodName.toString());
        this.log.debug("Finished {}", (Object)metricName);
    }

    private void sumMetric(String metricName, OffsetDateTime start, OffsetDateTime end, AnalyticsConstants.AggregationPeriod periodName) {
        CancelableHelper.checkCancellation();
        if (this.isMetricAlreadyRecorded(metricName, end, periodName)) {
            this.log.debug("Found persisted {} @ {} period {}", new Object[]{metricName, end, periodName});
            return;
        }
        this.log.debug("Summing {} for {} to {}", new Object[]{metricName, start, end});
        Long sum = this.metricStore.getSumMetricValue(metricName, start, end);
        this.log.debug("Found {} {}", (Object)sum, (Object)metricName);
        this.aggregatedMetricStore.addMetricData(metricName, sum, end, periodName.toString());
        this.log.debug("Finished {}", (Object)metricName);
    }

    private long processUniqueUserLogins(OffsetDateTime start, OffsetDateTime end, AnalyticsConstants.AggregationPeriod periodName) {
        CancelableHelper.checkCancellation();
        Optional<AggregatedMetric> metric = this.findRecordedMetric("unique_user_count", end, periodName);
        if (metric.isPresent()) {
            this.log.debug("Found persisted {} @ {} period {}", new Object[]{"unique_user_count", end, periodName});
            return metric.get().getMetricValue();
        }
        this.log.debug("Computing unique logins for {} to {}", (Object)start, (Object)end);
        Long count = this.loginStore.getUniqueLoginsCountForPeriod(start, end);
        this.log.debug("Found {} unique users", (Object)count);
        this.aggregatedMetricStore.addMetricData("unique_user_count", count, end, periodName.toString());
        this.log.debug("Finished unique logins");
        return count;
    }

    private void pruneLogins() {
        OffsetDateTime cutoff = this.aggregationTime.minusDays(30L);
        this.log.debug("Pruning login data before {}", (Object)cutoff);
        this.loginStore.deleteEntitiesBeforeDate(cutoff);
        this.log.debug("Completed pruning login data");
    }

    private void pruneMetricData(String metricName) {
        OffsetDateTime cutoff = this.aggregationTime.minusDays(30L);
        this.log.debug("Pruning {} before {}", (Object)metricName, (Object)cutoff);
        this.metricStore.deleteBeforeDate(metricName, cutoff);
        this.log.debug("Completed pruning {}", (Object)metricName);
    }

    private boolean isMetricAlreadyRecorded(String metricName, OffsetDateTime timestamp, AnalyticsConstants.AggregationPeriod period) {
        return this.findRecordedMetric(metricName, timestamp, period).isPresent();
    }

    private Optional<AggregatedMetric> findRecordedMetric(String metricName, OffsetDateTime timestamp, AnalyticsConstants.AggregationPeriod period) {
        String periodName = period.toString();
        return this.recentMetricsByName.getOrDefault(metricName, Collections.emptyList()).stream().filter(metric -> periodName.equals(metric.getPeriodType())).filter(metric -> metric.getMetricDate().equals(timestamp)).findAny();
    }

    @VisibleForTesting
    protected OffsetDateTime currentTime() {
        return OffsetDateTime.now();
    }

    private static boolean isDayStart(OffsetDateTime end) {
        return end.getHour() == 0 && end.getMinute() == 0;
    }

    private long getTotalComponents() {
        Iterable repositories = this.repositoryManager.browse();
        return this.commonQueryDataService.getComponentCount(repositories);
    }
}

