/*
 * Decompiled with CFR 0.152.
 */
package org.sonatype.nexus.ldap.internal.connector.dao;

import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.common.collect.Sets;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import javax.inject.Inject;
import javax.inject.Named;
import javax.inject.Singleton;
import javax.naming.InvalidNameException;
import javax.naming.NamingEnumeration;
import javax.naming.NamingException;
import javax.naming.directory.Attribute;
import javax.naming.directory.SearchControls;
import javax.naming.directory.SearchResult;
import javax.naming.ldap.LdapContext;
import javax.naming.ldap.LdapName;
import org.sonatype.goodies.common.ComponentSupport;
import org.sonatype.nexus.common.text.Strings2;
import org.sonatype.nexus.ldap.internal.connector.dao.LdapAuthConfiguration;
import org.sonatype.nexus.ldap.internal.connector.dao.LdapDAOException;
import org.sonatype.nexus.ldap.internal.connector.dao.LdapEncoder;
import org.sonatype.nexus.ldap.internal.connector.dao.LdapGroupDAO;
import org.sonatype.nexus.ldap.internal.connector.dao.LdapUser;
import org.sonatype.nexus.ldap.internal.connector.dao.LdapUserDAO;
import org.sonatype.nexus.ldap.internal.connector.dao.LdapUtils;
import org.sonatype.nexus.ldap.internal.connector.dao.NoLdapUserRolesFoundException;
import org.sonatype.nexus.ldap.internal.connector.dao.NoSuchLdapGroupException;
import org.sonatype.nexus.ldap.internal.connector.dao.NoSuchLdapUserException;

@Singleton
@Named
public class DefaultLdapGroupDAO
extends ComponentSupport
implements LdapGroupDAO {
    private final LdapUserDAO ldapUserManager;
    private final int pageSize;

    @Inject
    public DefaultLdapGroupDAO(LdapUserDAO ldapUserManager, @Named(value="${nexus.ldap.pageSize:-500}") @Named(value="${nexus.ldap.pageSize:-500}") int pageSize) {
        this.ldapUserManager = (LdapUserDAO)Preconditions.checkNotNull((Object)ldapUserManager);
        this.pageSize = pageSize;
    }

    private static boolean isGroupsEnabled(LdapAuthConfiguration configuration) {
        return configuration.isLdapGroupsAsRoles();
    }

    @Override
    public Set<String> getGroupMembership(String username, LdapContext context, LdapAuthConfiguration configuration) throws LdapDAOException, NoLdapUserRolesFoundException {
        boolean dynamicGroups = !Strings2.isEmpty((String)configuration.getUserMemberOfAttribute());
        boolean groupsEnabled = DefaultLdapGroupDAO.isGroupsEnabled(configuration);
        Set<String> roleIds = new HashSet<String>();
        if (groupsEnabled) {
            if (!dynamicGroups) {
                roleIds = this.getGroupMembershipFromGroups(username, context, configuration);
            } else {
                try {
                    roleIds = this.getGroupMembershipFromUser(username, context, configuration);
                }
                catch (NoSuchLdapUserException noSuchLdapUserException) {
                    throw new NoLdapUserRolesFoundException(username);
                }
            }
            if (roleIds == null || roleIds.isEmpty()) {
                throw new NoLdapUserRolesFoundException(username);
            }
        } else if (dynamicGroups && !groupsEnabled) {
            throw new NoLdapUserRolesFoundException(username);
        }
        return roleIds;
    }

    @Override
    public Set<String> getAllGroups(LdapContext context, LdapAuthConfiguration configuration) throws LdapDAOException {
        Set<Object> groups = Sets.newHashSet();
        if (DefaultLdapGroupDAO.isGroupsEnabled(configuration)) {
            try {
                if (Strings2.isEmpty((String)configuration.getUserMemberOfAttribute())) {
                    String groupIdAttribute = configuration.getGroupIdAttribute();
                    String groupBaseDn = Strings.nullToEmpty((String)configuration.getGroupBaseDn());
                    String filter = "(objectClass=" + configuration.getGroupObjectClass() + ")";
                    this.log.debug("Searching for groups in group DN: " + groupBaseDn + "\nUsing filter: '" + filter + "'");
                    SearchControls ctls = this.getBaseSearchControls(new String[]{groupIdAttribute}, configuration.isGroupSubtree());
                    List<SearchResult> results = LdapUtils.pagedLdapSearch(context, this.pageSize, () -> context.search(groupBaseDn, filter, ctls));
                    groups = this.getGroupIdsFromSearch(results, groupIdAttribute, configuration);
                } else {
                    String memberOfAttribute = configuration.getUserMemberOfAttribute();
                    String filter = "(objectClass=" + configuration.getUserObjectClass() + ")";
                    SearchControls ctls = this.getBaseSearchControls(new String[]{memberOfAttribute}, true);
                    String userBaseDn = Strings.nullToEmpty((String)configuration.getUserBaseDn());
                    List<SearchResult> results = LdapUtils.pagedLdapSearch(context, this.pageSize, () -> context.search(userBaseDn, filter, ctls));
                    Set<String> roles = this.getGroupIdsFromSearch(results, memberOfAttribute, configuration);
                    for (String groupDN : roles) {
                        groups.add(this.getGroupFromString(groupDN));
                    }
                }
            }
            catch (IOException | NamingException e) {
                String message = "Failed to get list of groups.";
                throw new LdapDAOException(message, e);
            }
        }
        return groups;
    }

    private SearchControls getBaseSearchControls(String[] returningAttributes, boolean subtree) {
        SearchControls ctls = new SearchControls();
        ctls.setReturningAttributes(returningAttributes);
        ctls.setSearchScope(subtree ? 2 : 1);
        return ctls;
    }

    private Set<String> getGroupIdsFromSearch(List<SearchResult> searchResults, String groupIdAttribute, LdapAuthConfiguration configuration) throws NamingException {
        LinkedHashSet<String> roles = new LinkedHashSet<String>();
        Map<String, Set<String>> mappings = configuration.getGroupReverseMappings();
        for (SearchResult result : searchResults) {
            Attribute groupIdAttr = result.getAttributes().get(groupIdAttribute);
            if (groupIdAttr == null) continue;
            NamingEnumeration<?> attributes = groupIdAttr.getAll();
            while (attributes.hasMoreElements()) {
                String group = String.valueOf(attributes.nextElement());
                Set<String> mappedRoles = mappings.get(group);
                if (mappedRoles == null) {
                    roles.add(group);
                    continue;
                }
                roles.addAll(mappedRoles);
            }
        }
        return roles;
    }

    private Set<String> getGroupMembershipFromUser(String username, LdapContext context, LdapAuthConfiguration configuration) throws LdapDAOException, NoSuchLdapUserException {
        LdapUser user = this.ldapUserManager.getUser(username, context, configuration);
        return Collections.unmodifiableSet(user.getMembership());
    }

    private Set<String> getGroupMembershipFromGroups(String username, LdapContext context, LdapAuthConfiguration configuration) throws LdapDAOException {
        String groupIdAttribute = configuration.getGroupIdAttribute();
        String groupMemberAttribute = configuration.getGroupMemberAttribute();
        String groupBaseDn = Strings.nullToEmpty((String)configuration.getGroupBaseDn());
        String groupMemberFormat = configuration.getGroupMemberFormat();
        String filter = "(&(objectClass={0})(&({1}=*)(";
        ArrayList<String> filterValues = new ArrayList<String>();
        filterValues.add(configuration.getGroupObjectClass());
        filterValues.add(groupIdAttribute);
        if (groupMemberFormat != null) {
            String member = groupMemberFormat.replace("${username}", "{2}");
            if (groupMemberFormat.contains("${username}")) {
                filterValues.add(LdapEncoder.nameEncode(username));
            }
            if (groupMemberFormat.contains("${dn}")) {
                LdapUser user;
                try {
                    user = this.ldapUserManager.getUser(username, context, configuration);
                }
                catch (NoSuchLdapUserException e) {
                    String message = "Failed to retrieve role information from ldap for user: " + username;
                    throw new LdapDAOException(message, e);
                }
                member = member.replace("${dn}", "{" + filterValues.size() + "}");
                filterValues.add(user.getDn());
            }
            filter = String.valueOf(filter) + groupMemberAttribute + "=" + member + ")))";
        } else {
            filterValues.add(LdapEncoder.nameEncode(username));
            filter = String.valueOf(filter) + groupMemberAttribute + "={2})))";
        }
        this.log.debug("Searching for group membership of: " + username + " in group DN: " + groupBaseDn + "\nUsing filter: '" + filter + "'");
        try {
            SearchControls ctls = this.getBaseSearchControls(new String[]{groupIdAttribute}, configuration.isGroupSubtree());
            String filterCopy = filter;
            List<SearchResult> results = LdapUtils.pagedLdapSearch(context, this.pageSize, () -> context.search(groupBaseDn, filterCopy, filterValues.toArray(), ctls));
            return this.getGroupIdsFromSearch(results, groupIdAttribute, configuration);
        }
        catch (IOException | NamingException e) {
            String message = "Failed to retrieve role information from ldap for user: " + username;
            throw new LdapDAOException(message, e);
        }
    }

    private String getGroupFromString(String dnString) {
        String result = dnString;
        try {
            LdapName dn = new LdapName(dnString);
            result = String.valueOf(dn.getRdn(dn.size() - 1).getValue());
        }
        catch (InvalidNameException invalidNameException) {
            this.log.debug("Expected a Group DN but found: " + dnString);
        }
        return result;
    }

    @Override
    public String getGroupName(String groupId, LdapContext context, LdapAuthConfiguration configuration) throws LdapDAOException, NoSuchLdapGroupException {
        if (!DefaultLdapGroupDAO.isGroupsEnabled(configuration)) {
            throw new NoSuchLdapGroupException(groupId, groupId);
        }
        if (Strings2.isEmpty((String)configuration.getUserMemberOfAttribute())) {
            Set<String> groups;
            String groupIdAttribute = configuration.getGroupIdAttribute();
            String groupBaseDn = Strings.nullToEmpty((String)configuration.getGroupBaseDn());
            String filter = "(&(objectClass=" + configuration.getGroupObjectClass() + ") (" + groupIdAttribute + "=" + groupId + "))";
            SearchControls ctls = this.getBaseSearchControls(new String[]{groupIdAttribute}, configuration.isGroupSubtree());
            try {
                List<SearchResult> results = LdapUtils.pagedLdapSearch(context, this.pageSize, () -> context.search(groupBaseDn, filter, ctls));
                groups = this.getGroupIdsFromSearch(results, groupIdAttribute, configuration);
            }
            catch (IOException | NamingException e) {
                throw new LdapDAOException("Failed to find group: " + groupId, e);
            }
            if (groups.isEmpty()) {
                throw new NoSuchLdapGroupException(groupId, groupId);
            }
            if (groups.size() > 1) {
                throw new NoSuchLdapGroupException(groupId, "More then one group found for group: " + groupId);
            }
            return groups.iterator().next();
        }
        String memberOfAttribute = configuration.getUserMemberOfAttribute();
        String filter = "(objectClass=" + configuration.getUserObjectClass() + ")";
        SearchControls ctls = this.getBaseSearchControls(new String[]{memberOfAttribute}, true);
        String userBaseDn = Strings.nullToEmpty((String)configuration.getUserBaseDn());
        try {
            List<SearchResult> results = LdapUtils.pagedLdapSearch(context, this.pageSize, () -> context.search(userBaseDn, filter, ctls));
            Set<String> roles = this.getGroupIdsFromSearch(results, memberOfAttribute, configuration);
            for (String groupDN : roles) {
                if (!groupId.equals(this.getGroupFromString(groupDN))) continue;
                return groupId;
            }
        }
        catch (IOException | NamingException e) {
            throw new LdapDAOException("Failed to find group: " + groupId, e);
        }
        throw new NoSuchLdapGroupException(groupId, groupId);
    }
}

