(function () {
    var scheduler = definePackage("scheduler");

    function shouldExecute(options) {
        if (isNull(options)) {
            return true;
        }
        if (isNull(options.executionCondition)) {
            return true;
        }
        if (isFunction(options.executionCondition)) {
            return options.executionCondition();
        }
        return Boolean(options.executionCondition);
    }

    function handleError(error, options) {
        if (isNull(options)) {
            console.error(error);
            return;
        }
        if (isFunction(options.errorHandler)) {
            options.errorHandler(error);
            return;
        }
        console.error(error);
    }

    function executeJob(jobData) {
        if (isNull(jobData.count)) {
            jobData.count = 0;
        }
        if (isNotNull(jobData.maxExecutions)) {
            if (jobData.count >= jobData.maxExecutions) {
                return;
            }
        }
        if (jobData.count == 0) {
            setTimeout(() => {
                if (!shouldExecute(jobData.options)) {
                    return;
                }
                jobData.count += 1;
                try {
                    jobData.exec();
                }
                catch (error) {
                    handleError(error, jobData.options);
                }
                executeJob(jobData);
            }, jobData.initialDelay);
        }
        else {
            setTimeout(() => {
                if (!shouldExecute(jobData.options)) {
                    return;
                }
                jobData.count += 1;
                try {
                    jobData.exec();
                }
                catch (error) {
                    handleError(error, jobData.options);
                }
                executeJob(jobData);
            }, jobData.delay);
        }
    }

    scheduler.executeAt = function(dateOfExecution, exec, options) {
        if (isNull(dateOfExecution) || (!isNumber(dateOfExecution) && !isFunction(dateOfExecution.getTime))) {
            throw new Error("Date of execution parameter needs to be a number or date");
        }
        if (isNull(exec) || !isFunction(exec)) {
            throw new Error("Exec parameter needs to be a function");
        }
        let waitTime = 0;
        if (isNumber(dateOfExecution)) {
            waitTime = dateOfExecution - new Date().getTime();
        }
        else {
            waitTime = dateOfExecution.getTime() - new Date().getTime();
        }
        if (waitTime < 0) {
            waitTime = 0;
        }
        if (isNull(options)) {
            options = {};
        }
        let jobData = {
            count: 0,
            maxExecutions: 1,
            initialDelay: waitTime,
            delay: 0,
            exec: exec,
            options: options
        };
        executeJob(jobData);
        return {
            cancel: () => {
                options.executionCondition = false;
            },
            getExecutionCounter: () => {
                return jobData.count;
            }
        };
    }

    scheduler.periodicExecution = function(timeInterval, exec, initialDelay, executionCount, options) {
        if (isNull(timeInterval) || !isNumber(timeInterval)) {
            throw new Error("Time interval parameter needs to be a function");
        }
        if (isNull(exec) || !isFunction(exec)) {
            throw new Error("Exec parameter needs to be a function");
        }
        if (isNull(initialDelay) || initialDelay < 0) {
            initialDelay = 0;
        }
        if (isNull(options)) {
            options = {};
        }
        let jobData = {
            count: 0,
            maxExecutions: executionCount,
            initialDelay: initialDelay,
            delay: timeInterval,
            exec: exec,
            options: options
        };
        executeJob(jobData);
        return {
            cancel: () => {
                options.executionCondition = false;
            },
            getExecutionCounter: () => {
                return jobData.count;
            }
        };
    }

})();
