function isNull(obj) {
	return (obj === undefined || obj === null);
};

function isNotNull(obj) {
	return (obj !== undefined && obj !== null);
};

function isFunction(obj) {
	if (isNull(obj)) {
		return false;
	}
	var getType = {};
	return obj && getType.toString.call(obj) === '[object Function]';
};

function isArray(obj) {
	if (isNull(obj)) {
		return false;
	}
	return Object.prototype.toString.call(obj) === '[object Array]';
};

function isObject(obj) {
	if (isNull(obj)) {
		return false;
	}
	return (typeof obj === 'object');
};

function isString(obj) {
    if (isNull(obj)) {
		return false;
	}
	return (typeof obj === 'string');
}

function isNumber(obj) {
    if (isNull(obj)) {
		return false;
	}
	return (typeof obj === 'number');
}

function isValidNumber(val) {
	if (isTextEmpty(val)) {
		return false;
	}
	return val == Number(val);
};

function cloneObject(obj) {
    if (isNull(obj)) {
        return null;
    }
    if (isObject(obj)) {
        var copy = {};
        for (var key in obj) {
            copy[key] = obj[key];
        }
        return copy;
    }
    else if (isArray(obj)) {
        var copy = [];
        for (var key in obj) {
            copy[key] = obj[key];
        }
        return copy;
    }
    else {
    throw Error("Given object is not valid object type");
    }
};

function escapeHtml(text) {
	if (isNull(text)) {
		return null;
	}
	text = text.replace(new RegExp("&", 'g'), "&amp;");
	text = text.replace(new RegExp("<", 'g'), "&lt;");
	text = text.replace(new RegExp(">", 'g'), "&gt;");
	text = text.replace(new RegExp("\"", 'g'), "&quot;");
	text = text.replace(new RegExp("'", 'g'), "&#39;");
	text = text.replace(new RegExp("/", 'g'), "&#x2F;");
	return text;
};

function escapeRegExp(string) {
  return string.replace(/[.*+?^${}()|[\]\\]/g, "\\$&");
}


function isTextEmpty(text) {
	if (isNull(text)) {
		return true;
	}
	return text === "";
};

function trimText(text) {
	if (isTextEmpty(text)) {
		return "";
	}
	return String(text).trim();
};

function definePackage(packageName, target) {
    if (isTextEmpty(packageName)) {
        throw new Error("Package name is empty");
    }

    var packages = packageName.split(".");
    for (var subPackageName of packages) {
        if (isTextEmpty(subPackageName)) {
            throw new Error("Invalid package name: '" + packageName + "'");
        }
    }

    var currentPackage = target;
    if (isNull(currentPackage)) {
        currentPackage = window;
    }

    for (var subPackageName of packages) {
        if (isNull(currentPackage[subPackageName])) {
            currentPackage[subPackageName] = {};
            currentPackage[subPackageName].definePackage = function(name) {
                return definePackage(name, currentPackage[subPackageName]);
            };
        }
        currentPackage = currentPackage[subPackageName];
    }

    return currentPackage;
};

function defineModule(moduleName, module) {
    if (isTextEmpty(moduleName)) {
        throw new Error("Module name is empty");
    }
    if (isNull(module)) {
        throw new Error("Module not provided");
    }
    var packages = moduleName.split(".");
    if (packages.length == 1) {
        window[packages[0]] = module;
    }
    else {
        var pckg = definePackage(packages.slice(0, packages.length - 1).join("."));
        pckg[packages[packages.length - 1]] = module;
    }

    return module;
};

function loadModule(moduleName) {
    if (isTextEmpty(moduleName)) {
        throw new Error("Module name is empty");
    }

    var currentPackage = window;
    var packages = moduleName.split(".");
    for (var subPackageName of packages) {
        if (isTextEmpty(subPackageName)) {
            throw new Error("Invalid module name: '" + moduleName + "'");
        }
        if (isNull(currentPackage[subPackageName])) {
            throw new Error("Module '" + moduleName + "' not defined");
        }
        currentPackage = currentPackage[subPackageName];
    }

    if (isNull(currentPackage)) {
        throw new Error("Module '" + moduleName + "' not defined");
    }

    return currentPackage;
};
(function () {
    let restClient = definePackage("restClient");

    restClient.response = class RestResponse {

        constructor(status) {
            this.status = status;
        }

        isSuccessful() {
            if (isNull(this.status)) {
                return false;
            }
            return this.status >= 200 && this.status <= 299;
        }

        parseResponseHeaders(responseHeaders) {
            if (isTextEmpty(responseHeaders)) {
                return;
            }
            this.headers = {};
            for (let line of responseHeaders.split("\n")) {
                let separator = line.indexOf(":");
                if (separator < 0) {
                    continue;
                }
                let headerName = trimText(line.substring(0, separator));
                let headerValue = trimText(line.substring(separator + 1));
                this.headers[headerName] = headerValue;
            }
        }

        getContentType() {
            for (let headerName in this.headers) {
                if ("content-type" == headerName.toLowerCase()) {
                    return this.headers[headerName];
                }
            }
        }

        isTextResponse() {
            let responseContentType = this.getContentType();
            if (isNull(responseContentType)) {
                return false;
            }
            responseContentType = responseContentType.toLowerCase();
            return responseContentType.startsWith("text/") || responseContentType.startsWith("application/json") || responseContentType.startsWith("application/xml");
        }

        isJsonResponse() {
            let responseContentType = this.getContentType();
            if (isNull(responseContentType)) {
                return false;
            }
            responseContentType = responseContentType.toLowerCase();
            return responseContentType.startsWith("text/json") || responseContentType.startsWith("application/json");
        }

    };

    restClient.request = class RestRequest {

        constructor(method) {
            if (!isTextEmpty(method)) {
                this.method = method;
            }
            else {
                this.method = "GET";
            }
        }

        withParameters(parameters) {
            if (isNull(this.parameters)) {
                this.parameters = {};
            }
            for (let parameterName in parameters) {
                this.parameters[parameterName] = parameters[parameterName];
            }
            return this;
        }

        withParameter(parameterName, parameterValue) {
            if (isTextEmpty(parameterName)) {
                throw new Error("Parameter name not provided");
            }
            if (isNull(this.parameters)) {
                this.parameters = {};
            }
            this.parameters[parameterName] = parameterValue;
            return this;
        }

        withHeaders(headers) {
            if (isNull(this.headers)) {
                this.headers = {};
            }
            for (let headerName in headers) {
                this.headers[headerName] = headers[headerName];
            }
            return this;
        }

        withHeader(headerName, headerValue) {
            if (isTextEmpty(headerName)) {
                throw new Error("Header name not provided");
            }
            if (isNull(this.headers)) {
                this.headers = {};
            }
            this.headers[headerName] = headerValue;
            return this;
        }

        withBody(body) {
            this.body = body;
            return this;
        }

        withCache(cache) {
            this.cache = cache;
            return this;
        }

        getContentType() {
            for (let headerName in this.headers) {
                if ("content-type" == headerName.toLowerCase()) {
                    return this.headers[headerName];
                }
            }
        }

        execute(url, callback) {
            if (!isNull(callback) && !isFunction(callback)) {
                throw new Error("Callback parameter needs to be a function");
            }

            let targetUrl = contextPath + url;

            if (isNotNull(this.parameters)) {
                for (let parameterName in this.parameters) {
                    if (targetUrl.indexOf("?") >= 0) {
                        targetUrl += "&";
                    }
                    else {
                        targetUrl += "?";
                    }
                    targetUrl += parameterName + "=" + encodeURIComponent(this.parameters[parameterName]);
                }
            }

            let response = null;

            let httpRequest = new XMLHttpRequest();
            httpRequest.open(this.method, targetUrl, isNotNull(callback));
            if (isNotNull(this.headers)) {
                for (let headerName in this.headers) {
                    httpRequest.setRequestHeader(headerName, this.headers[headerName]);
                }
            }

            if (!this.cache) {
                httpRequest.setRequestHeader('Cache-Control', 'no-cache, no-store, max-age=0');
                httpRequest.setRequestHeader('Expires', 'Thu, 1 Jan 1970 00:00:00 GMT');
                httpRequest.setRequestHeader('Pragma', 'no-cache');
            }

            httpRequest.onreadystatechange = function() {
                if (this.readyState == 4) {
                    response = new restClient.response(this.status);
                    response.parseResponseHeaders(this.getAllResponseHeaders());

                    if (response.isTextResponse()) {
                        response.data = this.responseText;
                    }
                    else {
                        response.data = this.response;
                    }

                    if (this.status >= 200 && this.status <= 299) {
                        if (isNotNull(callback)) {
                            callback(response.data, response);
                        }
                    }
                    else {
                        response.errorText = this.responseText;
                        if (isNotNull(callback)) {
                            callback(response.data, response);
                        }
                    }
                }
            };

            if (isNotNull(this.body)) {
                httpRequest.send(this.body);
            }
            else {
                httpRequest.send();
            }

            if (isNull(callback)) {
                return response;
            }
        }

    };

    restClient.get = function(url, parameters, headers, callback) {
        if (!isNull(callback) && !isFunction(callback)) {
            throw new Error("Callback parameter needs to be a function");
        }
        let request = new restClient.request("GET");
        if (!isNull(parameters)) {
            request.withParameters(parameters);
        }
        if (!isNull(headers)) {
            request.withHeaders(headers);
        }
        if (isNull(callback)) {
            let response = request.execute(url);
            if (response.isSuccessful()) {
                return response.data;
            }
            else {
                throw Error("Invalid response code: " + response.status);
            }
        }
        else {
            request.execute(url, callback);
        }
    }

    restClient.getJson = function(url, parameters, headers, callback) {
        if (!isNull(callback) && !isFunction(callback)) {
            throw new Error("Callback parameter needs to be a function");
        }
        let request = new restClient.request("GET");
        if (!isNull(parameters)) {
            request.withParameters(parameters);
        }
        let allHeaders = {
            "Accept": "application/json"
        };
        if (!isNull(headers)) {
            for (let headerName in headers) {
                allHeaders[headerName] = headers[headerName];
            }
        }
        request.withHeaders(allHeaders);
        if (isNull(callback)) {
            let response = request.execute(url);
            if (response.isSuccessful()) {
                if (response.isJsonResponse())  {
                    response.data = JSON.parse(response.data);
                }
                return response.data;
            }
            else {
                throw Error("Invalid response code: " + response.status);
            }
        }
        else {
            request.execute(url, (data, response) => {
                if (response.isSuccessful() && response.isJsonResponse()) {
                    response.data = JSON.parse(response.data);
                }
                callback(response.data, response);
            });
        }
    }

    restClient.post = function(url, body, parameters, headers, callback) {
        if (!isNull(callback) && !isFunction(callback)) {
            throw new Error("Callback parameter needs to be a function");
        }
        let request = new restClient.request("POST");
        if (!isNull(parameters)) {
            request.withParameters(parameters);
        }
        if (!isNull(headers)) {
            request.withHeaders(headers);
        }
        if (!isNull(body)) {
            request.withBody(body);
        }
        if (isNull(callback)) {
            let response = request.execute(url);
            if (response.isSuccessful()) {
                return response.data;
            }
            else {
                throw Error("Invalid response code: " + response.status);
            }
        }
        else {
            request.execute(url, callback);
        }
    }

    restClient.postFormData = function(url, body, parameters, headers, callback) {
        if (!isNull(callback) && !isFunction(callback)) {
            throw new Error("Callback parameter needs to be a function");
        }
        let request = new restClient.request("POST");
        if (!isNull(parameters)) {
            request.withParameters(parameters);
        }
        if (!isNull(headers)) {
            request.withHeaders(headers);
        }
        request.withHeader("Content-Type", "multipart/form-data");
        if (!isNull(body)) {
            request.withBody(body);
        }
        if (isNull(callback)) {
            let response = request.execute(url);
            if (response.isSuccessful()) {
                return response.data;
            }
            else {
                throw Error("Invalid response code: " + response.status);
            }
        }
        else {
            request.execute(url, callback);
        }
    }

    restClient.postJson = function(url, body, parameters, headers, callback) {
        if (!isNull(callback) && !isFunction(callback)) {
            throw new Error("Callback parameter needs to be a function");
        }
        let request = new restClient.request("POST");
        if (!isNull(parameters)) {
            request.withParameters(parameters);
        }
        let allHeaders = {
            "Accept": "application/json"
        };
        if (!isNull(body)) {
            allHeaders["Content-Type"] = "application/json; charset=utf8";
        }
        if (!isNull(headers)) {
            for (let headerName in headers) {
                allHeaders[headerName] = headers[headerName];
            }
        }
        request.withHeaders(allHeaders);
        if (!isNull(body)) {
            request.withBody(JSON.stringify(body));
        }
        if (isNull(callback)) {
            let response = request.execute(url);
            if (response.isSuccessful()) {
                if (response.isJsonResponse())  {
                    response.data = JSON.parse(response.data);
                }
                return response.data;
            }
            else {
                throw Error("Invalid response code: " + response.status);
            }
        }
        else {
            request.execute(url, (data, response) => {
                if (response.isSuccessful() && response.isJsonResponse()) {
                    response.data = JSON.parse(response.data);
                }
                callback(response.data, response);
            });
        }
    }

    restClient.put = function(url, body, parameters, headers, callback) {
        if (!isNull(callback) && !isFunction(callback)) {
            throw new Error("Callback parameter needs to be a function");
        }
        let request = new restClient.request("PUT");
        if (!isNull(parameters)) {
            request.withParameters(parameters);
        }
        if (!isNull(headers)) {
            request.withHeaders(headers);
        }
        if (!isNull(body)) {
            request.withBody(body);
        }
        if (isNull(callback)) {
            let response = request.execute(url);
            if (response.isSuccessful()) {
                return response.data;
            }
            else {
                throw Error("Invalid response code: " + response.status);
            }
        }
        else {
            request.execute(url, callback);
        }
    }

    restClient.putJson = function(url, body, parameters, headers, callback) {
        if (!isNull(callback) && !isFunction(callback)) {
            throw new Error("Callback parameter needs to be a function");
        }
        let request = new restClient.request("PUT");
        if (!isNull(parameters)) {
            request.withParameters(parameters);
        }
        let allHeaders = {
            "Accept": "application/json"
        };
        if (!isNull(body)) {
            allHeaders["Content-Type"] = "application/json; charset=utf8";
        }
        if (!isNull(headers)) {
            for (let headerName in headers) {
                allHeaders[headerName] = headers[headerName];
            }
        }
        request.withHeaders(allHeaders);
        if (!isNull(body)) {
            request.withBody(JSON.stringify(body));
        }
        if (isNull(callback)) {
            let response = request.execute(url);
            if (response.isSuccessful()) {
                if (response.isJsonResponse())  {
                    response.data = JSON.parse(response.data);
                }
                return response.data;
            }
            else {
                throw Error("Invalid response code: " + response.status);
            }
        }
        else {
            request.execute(url, (data, response) => {
                if (response.isSuccessful() && response.isJsonResponse()) {
                    response.data = JSON.parse(response.data);
                }
                callback(response.data, response);
            });
        }
    }

    restClient.delete = function(url, parameters, headers, callback) {
        if (!isNull(callback) && !isFunction(callback)) {
            throw new Error("Callback parameter needs to be a function");
        }
        let request = new restClient.request("DELETE");
        if (!isNull(parameters)) {
            request.withParameters(parameters);
        }
        if (!isNull(headers)) {
            request.withHeaders(headers);
        }
        if (isNull(callback)) {
            let response = request.execute(url);
            if (response.isSuccessful()) {
                return response.data;
            }
            else {
                throw Error("Invalid response code: " + response.status);
            }
        }
        else {
            request.execute(url, callback);
        }
    }

})();
(function () {
    var scheduler = definePackage("scheduler");

    function shouldExecute(options) {
        if (isNull(options)) {
            return true;
        }
        if (isNull(options.executionCondition)) {
            return true;
        }
        if (isFunction(options.executionCondition)) {
            return options.executionCondition();
        }
        return Boolean(options.executionCondition);
    }

    function handleError(error, options) {
        if (isNull(options)) {
            console.error(error);
            return;
        }
        if (isFunction(options.errorHandler)) {
            options.errorHandler(error);
            return;
        }
        console.error(error);
    }

    function executeJob(jobData) {
        if (isNull(jobData.count)) {
            jobData.count = 0;
        }
        if (isNotNull(jobData.maxExecutions)) {
            if (jobData.count >= jobData.maxExecutions) {
                return;
            }
        }
        if (jobData.count == 0) {
            setTimeout(() => {
                if (!shouldExecute(jobData.options)) {
                    return;
                }
                jobData.count += 1;
                try {
                    jobData.exec();
                }
                catch (error) {
                    handleError(error, jobData.options);
                }
                executeJob(jobData);
            }, jobData.initialDelay);
        }
        else {
            setTimeout(() => {
                if (!shouldExecute(jobData.options)) {
                    return;
                }
                jobData.count += 1;
                try {
                    jobData.exec();
                }
                catch (error) {
                    handleError(error, jobData.options);
                }
                executeJob(jobData);
            }, jobData.delay);
        }
    }

    scheduler.executeAt = function(dateOfExecution, exec, options) {
        if (isNull(dateOfExecution) || (!isNumber(dateOfExecution) && !isFunction(dateOfExecution.getTime))) {
            throw new Error("Date of execution parameter needs to be a number or date");
        }
        if (isNull(exec) || !isFunction(exec)) {
            throw new Error("Exec parameter needs to be a function");
        }
        let waitTime = 0;
        if (isNumber(dateOfExecution)) {
            waitTime = dateOfExecution - new Date().getTime();
        }
        else {
            waitTime = dateOfExecution.getTime() - new Date().getTime();
        }
        if (waitTime < 0) {
            waitTime = 0;
        }
        if (isNull(options)) {
            options = {};
        }
        let jobData = {
            count: 0,
            maxExecutions: 1,
            initialDelay: waitTime,
            delay: 0,
            exec: exec,
            options: options
        };
        executeJob(jobData);
        return {
            cancel: () => {
                options.executionCondition = false;
            },
            getExecutionCounter: () => {
                return jobData.count;
            }
        };
    }

    scheduler.periodicExecution = function(timeInterval, exec, initialDelay, executionCount, options) {
        if (isNull(timeInterval) || !isNumber(timeInterval)) {
            throw new Error("Time interval parameter needs to be a function");
        }
        if (isNull(exec) || !isFunction(exec)) {
            throw new Error("Exec parameter needs to be a function");
        }
        if (isNull(initialDelay) || initialDelay < 0) {
            initialDelay = 0;
        }
        if (isNull(options)) {
            options = {};
        }
        let jobData = {
            count: 0,
            maxExecutions: executionCount,
            initialDelay: initialDelay,
            delay: timeInterval,
            exec: exec,
            options: options
        };
        executeJob(jobData);
        return {
            cancel: () => {
                options.executionCondition = false;
            },
            getExecutionCounter: () => {
                return jobData.count;
            }
        };
    }

})();
var highchartsColors = [ '#4572A7', '#AA4643', '#89A54E', '#80699B', '#3D96AE', '#DB843D', '#92A8CD', '#A47D7C', '#B5CA92' ];

$(document).ready(function() {
	$("[rel='tooltip']").tooltip();
	
	$(".deleteConfirmationButton").on("click",function(event) {
		event.preventDefault();
		var href = $(this).attr("href");
		var dialogTitle = $(this).attr("dialog-title");
		var dialogMessage = $(this).attr("dialog-message");
		dialog.showConfirmationRedirect(dialogTitle,dialogMessage,href);
		return false;
	});
	
	$(".chosen-select").each(function(index) {
		var width = $(this).attr("width");
		var options = {
			"search_contains": true
		};
		if (width) {
			options.width = width;
		}
		$(this).chosen(options);
	});
	
	$(".color-picker").each(function(index) {
		var field = this;
		var color = "#FFFFFF";
		
		if ($(field).prop("tagName") == "INPUT") {
			if ($(field).val() != "") {
				color = $(field).val();
			}
			$(field).ColorPicker({
				"color": color,
				onChange: function(hsb, hex, rgb, el) {
					$(field).val(hex);
				}
			});
		}
		else {
			var rgb = $(field).css("background-color");
			if (rgb != "transparent") {
				rgb = rgb.substring(4, rgb.length - 1).replace(/\s/g, "").split(",");
				color = "#";
				for (var i = 0; i < 3; i++) {
					if (Number(rgb[i]) < 10) {
						color += "0";
					}
					color += Number(rgb[i]).toString(16);
				}
			}
			$(field).ColorPicker({
				"color": color,
				onChange: function(hsb, hex, rgb, el) {
					$(field).css('backgroundColor', '#' + hex);
				}
			});
		}
	});

	$("input.date").each(function(index) {
		var dateFormat = $(this).attr("date-format");
		if (!dateFormat) {
			dateFormat = defaultDateFormat;
		}
		$(this).datepicker({
			"dateFormat": dateFormat
		});
	});
	
	$("input.datetime").each(function(index) {
		var dateFormat = $(this).attr("date-format");
		var timeFormat = $(this).attr("time-format");
		if (!dateFormat) {
			dateFormat = defaultDateFormat;
		}
		if (!timeFormat) {
			timeFormat = defaultTimeFormat;
		}
		$(this).datetimepicker({
			"dateFormat": dateFormat,
			"timeFormat": timeFormat
		});
	});
	
	$("input.time").each(function(index) {
		var timeFormat = $(this).attr("time-format");
		if (!timeFormat) {
			timeFormat = defaultTimeFormat;
		}
		$(this).timepicker({
			"timeFormat": timeFormat
		});
	});

	$("[slider-for]").each(function(index) {
		var sliderField = this;
		var forInputName = $(sliderField).attr("slider-for");
		var forInput = $(sliderField).parent().find("input[name='" + forInputName + "']");
		var min = Number($(sliderField).attr("min"));
		var max = Number($(sliderField).attr("max"));
		var step = Number($(sliderField).attr("step"));

		if (!isValidNumber(min)) {
			min = 0;
		}
		if (!isValidNumber(max)) {
			max = 100;
		}
		if (!isValidNumber(step)) {
			step = 1;
		}

		$(sliderField).slider({
			"min": min,
			"max": max,
			"step": step,
			"value": $(forInput).val(),
			"change": function(event,ui) {
				forInput.val(ui.value);
			},
			"slide": function(event,ui) {
				forInput.val(ui.value);
			}
		});

		forInput.on("change",function() {
			var value = Number($(this).val());
			if ($(this).val() == "" || !isValidNumber(value)) {
				$(this).val("0");
			}
			if (value < min) {
				value = min;
			}
			if (value > max) {
				value = max;
			}
			$(sliderField).slider("value", value);
		});
		forInput.on("keyup", function() {
			var value = Number($(this).val());
			if ($(this).val() != "" && isValidNumber(value)) {
				if (value >= min && value <= max) {
					$(sliderField).slider("value",value);
				}
			}
		});
	});

	$("input.spinner").each(function(index) {
		var min = Number($(this).attr("min"));
		var max = Number($(this).attr("max"));
		var step = Number($(this).attr("step"));
		var numberFormat = $(this).attr("number-format");

		if (!isValidNumber(min)) {
			min = 0;
		}
		if (!isValidNumber(max)) {
			max = 100;
		}
		if (!isValidNumber(step)) {
			step = 1;
		}
		if (!numberFormat) {
			numberFormat = "n";
		}

		$(this).spinner({
			"min": min,
			"max": max,
			"step": step,
			"numberFormat": numberFormat
		});
	});
	
	$("#search").change(function(e) {
		e.preventDefault();
		$("#page").val("1");
		$("#page").parent().submit();
		return false;
	});
	
	$("#showPerPage").change(function(e) {
		e.preventDefault();
		$("#page").val("1");
		$("#page").parent().submit();
		return false;
	});
	
	$("ul.pagination a").on("click",function(event) {
		var selectedPageNum = $(this).attr("href").substring(1);
		var currentPageNum = Number($("#page").val());
		if (selectedPageNum == "prev") {
			currentPageNum -= 1;
			if (currentPageNum < 1) {
				currentPageNum = 1;
			}
		}
		else if (selectedPageNum == "next") {
			currentPageNum += 1;
		}
		else {
			currentPageNum = Number(selectedPageNum);
		}
		$("#page").val(String(currentPageNum));
		
		$("#page").parent().submit();
		
		return false;
	});
});
var DialogButton = Backbone.Model.extend({
	defaults: {
		"text": "",
		"click": null,
		"class": "btn btn-primary",
		"close": false
	},
	initialize: function() {
		
	}
});

var DialogButtonList = Backbone.Collection.extend({
	model: DialogButton
});

var DialogData = Backbone.Model.extend({
	defaults: {
		"type": "info",
		"title": "",
		"content": ""
	},
	initialize:function() {
		this.set({ buttons: new DialogButtonList() }, { silent: true });
	}
});

var DialogButtonView = Backbone.View.extend({
	'tagName': "button",
	'initialize': function() {
		
		
	},
	'render': function() {
		var obj = this;
		this.$el.text(this.model.get("text"));
		this.$el.attr("type", "button");
		this.$el.attr("class", this.model.get("class"));
		
		if (this.model.get("close")) {
			this.$el.attr("data-dismiss", "modal");
		}
		
		this.$el.on("click",function() {
			var func = obj.model.get("click");
			if (func != null) {
				func();
			}
		});
		
		return this;
	}
});

var DialogView = Backbone.View.extend({
	'tagName':"div",
	'template': "",
	'initialize': function() {
		this.template = templates.getTemplate("/templates/dialog");
		this.$el.attr("class","modal fade");
	},
	show:function() {
		var obj = this;
		
		$("body").append(this.el);
		
		this.$el.modal({
	    	show: false
	    });
	    
		this.$el.on("hidden.bs.modal",function () {
			obj.$el.remove();
		});
		
		this.render();
		this.$el.modal("show");
	},
	hide: function() {
		
	},
	'render': function() {
		var obj = this;
		var data = {
			title: obj.model.get("title")
		};
		this.$el.html(Mustache.render(this.template,data));
		this.$el.find("div.dialogContent").html(this.model.get("content"));
		
		//add buttons
		for (var i = 0; i < this.model.get("buttons").length; i++) {
			var button = this.model.get("buttons").at(i);
			var dialogButtonView = new DialogButtonView({ model: button });
			this.$el.find(".modal-footer").append(dialogButtonView.render().el);
		}
		
		return this;
	}
});

var dialog = {
	showDialog: function(title,content,buttons,openFunction) {
		var dialogData=new DialogData({
			type:"info",
			title:title,
			content:content
		});
		for (var i=0;i<buttons.length;i++) {
			var button=buttons[i];
			dialogData.get("buttons").add(new DialogButton({
				"text":button.text,
				"class":button.class,
				"close":button.close,
				"click":button.click
			}));
		}
		
		var dialogView=new DialogView({
			model:dialogData
		});
		
		if (openFunction) {
			$(dialogView.el).on("show",function () {
				openFunction(dialogView.el);
			});
		}
		
		dialogView.show();
		
		return dialogView.el;
	},
	showInfoMessage: function(message,title) {
		var dialogTitle=getSpringMessage("common.title.info");
		if (title) {
			dialogTitle=title;
		}
		
		var dialogData=new DialogData({
			type:"info",
			title:dialogTitle,
			content:message
		});
		
		var dialogView=new DialogView({
			model:dialogData
		});
		
		dialogData.get("buttons").add(new DialogButton({
			"text":getSpringMessage("common.close"),
			"class":"btn btn-primary",
			"close":true
		}));
		
		dialogView.show();
	},
	showErrorMessage: function(message) {
		var dialogTitle=getSpringMessage("common.title.error");
		if (title) {
			dialogTitle=title;
		}
		
		var dialogData=new DialogData({
			type:"info",
			title:dialogTitle,
			content:message
		});
		
		var dialogView=new DialogView({
			model:dialogData
		});
		
		dialogData.get("buttons").add(new DialogButton({
			"text":getSpringMessage("common.close"),
			"class":"btn btn-primary",
			"close":true
		}));
		
		dialogView.show();
	},
	showPopup: function(item,title,content) {
		$(item).popover({
			trigger:"hover",
			title:title,
			content:content
		});
		$(item).popover('show');
	},
	showConfirmation: function(title,text,confirmationAction) {
		var dialogData=new DialogData({
			type:"info",
			title:title,
			content:text
		});
		
		var dialogView=new DialogView({
			model:dialogData
		});
		
		dialogData.get("buttons").add(new DialogButton({
			"text":getSpringMessage("common.confirm"),
			"class":"btn btn-primary",
			"close":false,
			"click":function() {
				$(dialogView.el).modal("hide");
				confirmationAction();
			}
		}));
		
		dialogData.get("buttons").add(new DialogButton({
			"text":getSpringMessage("common.cancel"),
			"class":"btn btn-danger",
			"close":true
		}));
		
		dialogView.show();
	},
	showConfirmationRedirect: function(title,text,url) {
		dialog.showConfirmation(title,text,function() {
			window.location=url;
		});
	}
};
$(document).ready(function() {
	$(".pagination a").on("click",function(event) {
		event.preventDefault();
		
		var formId=$(this).parent().parent().attr("formId");
		if ($(this).attr("href")!=undefined) {
			var page=$(this).attr("href").substring(1);
			
			var currentPage=Number($("#"+formId+" input[name='page']").val());
			
			if (page=="prev") {
				currentPage-=1;
			}
			else if (page=="next") {
				currentPage+=1;
			}
			else {
				currentPage=Number(page);
			}
			$("#"+formId+" input[name='page']").val(currentPage);
			$("#"+formId).submit();
		}
	});
	
	$(".pagination input").on("change",function(event) {
		var formId=$(this).parent().parent().attr("formId");
		$("#"+formId+" input[name='page']").val($(this).val());
		$("#"+formId).submit();
	});
});var templates = {
    cache: {},
    getTemplate: function(path) {
        if (isNotNull(this.cache[path])) {
            return this.cache[path];
        }
        var obj = this;
        $.ajax({
            async: false,
            url: contextPath + path,
            type: 'GET',
            success: function(responseData) {
                obj.cache[path] = responseData;
            }
        });
        return this.cache[path];
    }
};(function () {
	var router = definePackage("router");

    router.enabled = false;
	router.routes = [];

    window.addEventListener('hashchange', function (event){
        if (!router.enabled) {
            return;
        }
        var url = event.newURL;
        var n = url.indexOf("#");
        if (n >= 0) {
            url = url.substring(n + 1);
            router.handleLocationChange(url);
        }
    });

    function matchesRouteUrl(routeUrl, url) {
        if (routeUrl == url) {
            return true;
        }
        if (routeUrl.endsWith("*")) {
            var prefix = routeUrl.substring(0, routeUrl.length - 1);
            if (url.startsWith(prefix)) {
                return true;
            }
        }
        return false;
    };

    router.handleLocationChange = function(url) {
        for (var route of router.routes) {
            var matches = false;
            if (isString(route.target)) {
                matches = matchesRouteUrl(route.target, url);
            }
            else if (isArray(route.target)) {
                var urlArray = this.decodeUrl(url);
                for (var i in route.target) {
                    var item = route.target[i];
                    var urlPart = urlArray[i];
                    if (isNull(urlPart)) {
                        continue;
                    }
                    if (!isString(item)) {
                        throw new Error("Route item is not string");
                    }
                    matches = matchesRouteUrl(item, urlPart);
                    if (matches) {
                        break;
                    }
                }
            }
            else {
                throw new Error("Target is not string or array");
            }
            if (matches) {
                route.callback(url);
                return;
            }
        }
        if (isFunction(router.unknownRouteHandler)) {
            router.unknownRouteHandler(url);
        }
    }

    router.changeLocation = function(url) {
        if (isString(url)) {
            window.location.hash = url;
        }
        else if (isArray(url)) {
            window.location.hash = router.encodeUrl(url);
        }
        else {
            throw new Error("Url is not string or array");
        }
    };

    router.handleRoute = function(target, callback) {
        if (isNull(target)) {
            throw new Error("Target not provided");
        }
        if (!isString(target) && !isArray(target)) {
            throw new Error("Target is not string or array");
        }
        if (!isFunction(callback)) {
            throw new Error("Callback is not a function");
        }
        router.routes.push({
            "target": target,
            "callback": callback
        });
    };

    router.handleUnknownRoute = function(callback) {
        if (!isFunction(callback)) {
            throw new Error("Callback is not a function");
        }
        router.unknownRouteHandler = callback;
    }

    router.encodeUrl = function(url) {
        if (!isArray(url)) {
            throw new Error("Url is not array");
        }
        var target = "";
        for (var item of url) {
            if (target.length > 0) {
                target += "/";
            }
            target += encodeURIComponent(item);
        }
        return target;
    };

    router.decodeUrl = function(url) {
        if (!isString(url)) {
            throw new Error("Url is not string");
        }
        var parts = url.split("/");
        var target = [];
        for (var item of parts) {
            target.push(decodeURIComponent(item));
        }
        return target;
    };

})();

$(document).ready(function() {
    setTimeout(() => {
        var router = loadModule("router");
        if (!router.enabled) {
            return;
        }
        var url = window.location.href;
        var n = url.indexOf("#");
        if (n >= 0) {
            url = url.substring(n + 1);
            router.handleLocationChange(url);
        }
        else {
            router.handleLocationChange("");
        }
    }, 50);
});(function() {
    definePackage("jtekma");
    let view = definePackage("jtekma.view");

    view.main = class MainView {

        constructor(tag, template, model, attributes, events) {
            if (isTextEmpty(tag)) {
                throw new Error("Tag needs to be provided");
            }
            this.el = document.createElement(tag);
            this.tag = tag;

            if (isNotNull(template) && !isString(template) && !isFunction(template)) {
                throw new Error("Template can only be string or function");
            }
            else {
                this.template = template;
            }

            if (!isNull(model)) {
                this.model = model;
            }
            else {
                this.model = {};
            }
            if (isNotNull(attributes)) {
                this.attributes = attributes;
            }
            else {
                this.attributes = {};
            }
            if (isNotNull(events)) {
                this.events = events;
            }
            else {
                this.events = {};
            }
        }

        setContent(content) {
            if (isString(content)) {
                this.setContentText(content);
            }
            else {
                this.content = content;
            }
            if (this.drawn) {
                this.draw();
            }
        }

        setContentText(content) {
            if (!isTextEmpty(content)) {
                this.content = escapeHtml(String(content));
            }
            else {
                this.content = "";
            }
            if (this.drawn) {
                this.draw();
            }
        }

        setContentHtml(content) {
            if (!isTextEmpty(content)) {
                this.content = content;
            }
            else {
                this.content = "";
            }
            if (this.drawn) {
                this.draw();
            }
        }

        addContent(item) {
            if (isNull(item)) {
                return;
            }
            if (isNull(this.content)) {
                this.content = [];
            }
            if (!isArray(this.content)) {
                let c = this.content;
                this.content = [c];
            }
            if (isArray(item)) {
                for (let i of item) {
                    this.content.push(i);
                }
            }
            else {
                this.content.push(item);
            }
            if (this.drawn) {
                this.draw();
            }
        }

        removeContent(item) {
            if (isNull(item)) {
                return;
            }
            if (isArray(item)) {
                for (let i of item) {
                    if (this.content == i) {
                        this.content = [];
                    }
                    else if (isArray(this.content)) {
                        this.content = this.content.filter((el) => {
                            return el !== i;
                        });
                    }
                }
            }
            else {
                if (this.content == item) {
                    this.content = [];
                }
                else if (isArray(this.content)) {
                    this.content = this.content.filter((el) => {
                        return el !== item;
                    });
                }
            }
            if (this.drawn) {
                this.draw();
            }
        }

        addTextContent(text) {
            this.addContent(escapeHtml(text));
        }

        clearContent() {
            this.setContentHtml("");
        }

        setId(id) {
            this.setAttribute("id", id);
        }

        setClass(clazz) {
            this.setAttribute("class", clazz);
        }

        addClass(clazz) {
            if (isTextEmpty(this.attributes.class)) {
                this.setAttribute("class", clazz);
            }
            else {
                let updatedClass = "";
                let classAlreadyPresent = false;
                for (let p of this.attributes.class.split(" ")) {
                    if (isTextEmpty(p)) {
                        continue;
                    }
                    if (p == clazz) {
                        classAlreadyPresent = true;
                    }
                    if (updatedClass.length > 0) {
                        updatedClass += " ";
                    }
                    updatedClass += p;
                }
                if (!classAlreadyPresent) {
                    if (updatedClass.length > 0) {
                        updatedClass += " ";
                    }
                    updatedClass += clazz;
                }
                this.setAttribute("class", updatedClass);
            }
        }

        removeClass(clazz) {
            if (isTextEmpty(this.attributes.class)) {
                return;
            }
            let updatedClass = "";
            for (let p of this.attributes.class.split(" ")) {
                if (isTextEmpty(p)) {
                    continue;
                }
                if (p == clazz) {
                    continue;
                }
                if (updatedClass.length > 0) {
                    updatedClass += " ";
                }
                updatedClass += p;
            }
            this.setAttribute("class", updatedClass);
        }

        setStyle(style) {
            this.setAttribute("style", style);
        }

        addStyle(style, value) {
            let styleKey = style;
            if (isNull(value)) {
                let parts = style.split(":");
                if (parts.length > 1) {
                    styleKey = parts[0];
                    value = parts[1];
                }
            }
            if (isTextEmpty(this.attributes.style)) {
                this.setAttribute("style", styleKey + ": " + value);
            }
            else {
                let updatedStyle = "";
                let parts = this.attributes.style.split(";");
                let styleOverridden = false;
                for (let p of parts) {
                    if (isTextEmpty(p)) {
                        continue;
                    }
                    let t = p.split(":");
                    if (t.length > 1) {
                        if (t[0].trim() == styleKey.trim()) {
                            styleOverridden = true;
                            updatedStyle += styleKey.trim() + ": " + value.trim() + ";";
                        }
                        else {
                            updatedStyle += p + ";";
                        }
                    }
                    else {
                        updatedStyle += p + ";";
                    }
                }
                if (!styleOverridden) {
                    updatedStyle += styleKey.trim() + ": " + value.trim() + ";";
                }
                this.setAttribute("style", updatedStyle);
            }
            if (!this.attributes.style.endsWith(";")) {
                this.setAttribute("style", this.attributes.style + ";");
            }
        }

        removeStyle(style) {
            if (isTextEmpty(this.attributes.style)) {
                return;
            }
            let updatedStyle = "";
            for (let p of this.attributes.style.split(";")) {
                if (isTextEmpty(p)) {
                    continue;
                }
                let t = p.split(":");
                if (t.length > 1) {
                    if (t[0].trim() != style.trim()) {
                        updatedStyle += p + ";";
                    }
                }
                else {
                    updatedStyle += p + ";";
                }
            }
            this.setAttribute("style", updatedStyle);
        }

        setAttribute(key, value) {
            if (isNull(this.attributes)) {
                this.attributes = {};
            }
            this.attributes[key] = value;
            if (this.drawn) {
                this.draw();
            }
        }

        removeAttribute(key) {
            if (isNotNull(this.attributes)) {
                this.attributes[key] = null;
            }
            if (this.drawn) {
                this.draw();
            }
        }

        onClick(callback) {
            this.addEventListener("click", callback);
        }

        addEventListener(eventType, callback) {
            if (!isFunction(callback)) {
                throw new Error("Callback parameter needs to be a function");
            }
            this.events[eventType] = callback;
            if (this.drawn) {
                this.draw();
            }
        }

        show() {
            this.removeStyle("display");
            if (this.drawn) {
                this.draw();
            }
        }

        hide() {
            this.addStyle("display", "none");
            if (this.drawn) {
                this.draw();
            }
        }

        render() {
            if (isNull(this.template)) {
                return null;
            }
            if (isFunction(this.preRender)) {
                this.preRender();
            }
            let t = null;
            if (isString(this.template)) {
                t = this.template;
            }
            else if (isFunction(this.template)) {
                t = this.template();
            }
            else {
                throw new Error("Template is not string or function");
            }
            if (isNull(t)) {
                return null;
            }
            let html = Mustache.render(t, this.model);
            if (isFunction(this.postRender)) {
                this.postRender();
            }
            return html;
        }

        draw(target) {
            let html = this.render();
            if (isFunction(this.preDraw)) {
                this.preDraw();
            }
            if (isNotNull(html)) {
                this.el.innerHTML = html;
            }
            else {
                if (isNull(this.content)) {
                    this.el.innerHTML = "";
                }
                else if (isString(this.content)) {
                    this.el.innerHTML = this.content;
                }
                else if (isArray(this.content)) {
                    this.el.innerHTML = "";
                    for (let item of this.content) {
                        if (isNull(item)) {
                            // ignore item
                        }
                        else if (isString(item)) {
                            this.el.append(item);
                        }
                        else if (isObject(item) && isFunction(item.draw)) {
                            item.draw(this.el);
                        }
                        else {
                            throw new Error("Content item for drawing is not supported");
                        }
                    }
                }
                else if (isObject(this.content) && isFunction(this.content.draw)) {
                    this.el.innerHTML = "";
                    this.content.draw(this.el);
                }
                else {
                    throw new Error("Content for drawing is not supported");
                }
            }
            if (isNotNull(this.attributes)) {
                for (let attributeKey in this.attributes) {
                    if (isNotNull(this.attributes[attributeKey])) {
                        this.el.setAttribute(attributeKey, this.attributes[attributeKey]);
                    }
                    else {
                        this.el.removeAttribute(attributeKey);
                    }
                }
            }
            if (isNotNull(this.events)) {
                for (let eventKey in this.events) {
                    if (isNull(this.events[eventKey])) {
                        continue;
                    }
                    if (!isFunction(this.events[eventKey])) {
                        throw new Error("Event for type '" + eventKey + "' is not function");
                    }
                    this.el.addEventListener(eventKey, this.events[eventKey]);
                }
            }
            if (isNotNull(target)) {
                target.append(this.el);
            }
            if (isFunction(this.postDraw)) {
                this.postDraw();
            }
            this.drawn = true;
        }

        isPresentInDom() {
            if (isNull(this.el)) {
                return false;
            }
            let parent = this.el.parentNode;
            while (isNotNull(parent)) {
                if (parent === document) {
                    return true;
                }
                parent = parent.parentNode;
            }
            return false;
        }
    };

})();(function() {
    let view = definePackage("jtekma.view");

    view.span = class Span extends view.main {

        constructor(content) {
            super("span");
            this.setContent(content);
        }

    };

})();(function() {
    let view = definePackage("jtekma.view");

    view.div = class Div extends view.main {

        constructor(content) {
            super("div");
            this.setContent(content);
        }

    };

})();(function() {
    let view = definePackage("jtekma.view");

    view.a = class Link extends view.main {

        constructor(content, href) {
            super("a");
            this.setContent(content);
            if (!isNull(href)) {
                this.attributes.href = href;
            }
        }

    };

})();(function() {
    let view = definePackage("jtekma.view");

    view.p = class Paragraph extends view.main {

        constructor(content) {
            super("p");
            this.setContent(content);
        }

    };

})();(function() {
    let view = definePackage("jtekma.view");

    view.b = class Bold extends view.main {

        constructor(content) {
            super("b");
            this.setContent(content);
        }

    };

})();(function() {
    let view = definePackage("jtekma.view");

    view.i = class Italic extends view.main {

        constructor(content) {
            super("i");
            this.setContent(content);
        }

    };

})();(function() {
    let view = definePackage("jtekma.view");

    view.br = class Br extends view.main {

        constructor() {
            super("br");
        }

    };

})();(function() {
    let view = definePackage("jtekma.view");

    view.hr = class Hr extends view.main {

        constructor() {
            super("hr");
        }

    };

})();(function() {
    let view = definePackage("jtekma.view");

    view.button = class Button extends view.main {

        constructor(content, clickEvent) {
            super("button");
            this.setContent(content);
            if (!isNull(clickEvent)) {
                this.events.click = clickEvent;
            }
        }

        setDisabled(disabled) {
            if (disabled) {
                this.setAttribute("disabled", disabled);
            }
            else {
                this.removeAttribute("disabled");
            }
        }

    };

})();(function() {
    let view = definePackage("jtekma.view");

    view.input = class Input extends view.main {

        constructor(type, value) {
            super("input");
            this.attributes.type = type;

            if (!isTextEmpty(value)) {
                this.attributes.value = value;
            }
        }

        getValue() {
            return this.el.value;
        }

        setValue(val) {
            this.el.value = val;
        }

        clear() {
            this.el.value = "";
        }

        setName(name) {
            this.setAttribute("name", name);
        }

        setDisabled(disabled) {
            if (disabled) {
                this.setAttribute("disabled", disabled);
            }
            else {
                this.removeAttribute("disabled");
            }
        }

        setReadonly(readonly) {
            if (readonly) {
                this.setAttribute("readonly", readonly);
            }
            else {
                this.removeAttribute("readonly");
            }
        }

        onChange(callback) {
            this.addEventListener("change", callback);
        }

    };

})();(function() {
    let view = definePackage("jtekma.view");

    view.table = class Table extends view.main {

        constructor(rows) {
            super("table");
            if (isArray(rows)) {
                if (rows.length > 0) {
                    this.setHeader(rows[0]);
                }
                if (rows.length > 1) {
                    this.setData(rows.splice(1, rows.length - 1));
                }
            }
        }

        setHeader(columns) {
            if (!isArray(columns)) {
                throw new Error("Provided columns is not array");
            }
            if (!isArray(this.content)) {
                this.content = [];
            }
            if (this.content.length < 1) {
                this.addContent(new view.thead([]));
            }
            if (!isObject(this.content[0]) || this.content[0].tag != "thead") {
                throw new Error("Table first row is not element with tag thead");
            }
            if (this.content[0].content.length < 1) {
                this.content[0].addContent(new view.tr([]));
            }
            if (!isObject(this.content[0].content[0]) || this.content[0].content[0].tag != "tr") {
                throw new Error("Table head first row is not element with tag tr");
            }
            this.content[0].content[0].setContent([]);
            for (let column of columns) {
                if (isNull(column)) {
                    this.content[0].content[0].addContent(new view.th(""));
                }
                else if (isString(column)) {
                    this.content[0].content[0].addContent(new view.th(column));
                }
                else if (isObject(column) && column.tag == "th") {
                    this.content[0].content[0].addContent(column);
                }
                else {
                    throw new Error("Unknown column data");
                }
            }
            if (this.drawn) {
                this.draw();
            }
        }

        setData(rows) {
            if (!isArray(rows)) {
                throw new Error("Provided rows is not array");
            }
            if (!isArray(this.content)) {
                this.content = [];
            }
            if (this.content.length < 1) {
                this.addContent(new view.thead([]));
            }
            if (this.content.length < 2) {
                this.addContent(new view.tbody([]));
            }
            this.content[1].content = [];
            if (!isObject(this.content[1]) || this.content[1].tag != "tbody") {
                throw new Error("Table second row is not element with tag tbody");
            }
            for (let row of rows) {
                if (!isArray(row)) {
                    throw new Error("Row data is not array");
                }
                let rowView = new view.tr([]);
                this.content[1].addContent(rowView);
                for (let column of row) {
                    if (isNull(column)) {
                        rowView.addContent(new view.td(""));
                    }
                    else if (isString(column)) {
                        rowView.addContent(new view.td(column));
                    }
                    else if (isObject(column) && column.tag == "td") {
                        rowView.addContent(column);
                    }
                    else {
                        throw new Error("Unknown column data");
                    }
                }
            }
            if (this.drawn) {
                this.draw();
            }
        }

        addData(columns) {
            if (!isArray(columns)) {
                throw new Error("Provided columns is not array");
            }
            if (!isArray(this.content)) {
                this.content = [];
            }
            if (this.content.length < 1) {
                this.addContent(new view.thead([]));
            }
            if (this.content.length < 2) {
                this.addContent(new view.tbody([]));
            }
            let rowView = new view.tr([]);
            this.content[1].addContent(rowView);
            for (let column of columns) {
                if (isNull(column)) {
                    rowView.addContent(new view.td(""));
                }
                else if (isString(column)) {
                    rowView.addContent(new view.td(column));
                }
                else if (isObject(column) && column.tag == "td") {
                    rowView.addContent(column);
                }
                else {
                    throw new Error("Unknown column data");
                }
            }
            if (this.drawn) {
                this.draw();
            }
        }

    };

})();(function() {
    let view = definePackage("jtekma.view");

    view.thead = class Thead extends view.main {

        constructor(columns) {
            super("thead");
            this.setContent(columns);
        }

    };

})();(function() {
    let view = definePackage("jtekma.view");

    view.tbody = class Tbody extends view.main {

        constructor(columns) {
            super("tbody");
            this.setContent(columns);
        }

    };

})();(function() {
    let view = definePackage("jtekma.view");

    view.td = class Td extends view.main {

        constructor(content) {
            super("td");
            this.setContent(content);
        }

    };

})();(function() {
    let view = definePackage("jtekma.view");

    view.th = class Th extends view.main {

        constructor(content) {
            super("th");
            this.setContent(content);
        }

    };

})();(function() {
    let view = definePackage("jtekma.view");

    view.tr = class Tr extends view.main {

        constructor(columns) {
            super("tr");
            this.setContent(columns);
        }

    };

})();(function() {
    let view = definePackage("jtekma.view");

    view.h1 = class Link extends view.main {

        constructor(content) {
            super("h1");
            this.setContent(content);
        }

    };

})();(function() {
    let view = definePackage("jtekma.view");

    view.h2 = class Link extends view.main {

        constructor(content) {
            super("h2");
            this.setContent(content);
        }

    };

})();(function() {
    let view = definePackage("jtekma.view");

    view.h3 = class Link extends view.main {

        constructor(content) {
            super("h3");
            this.setContent(content);
        }

    };

})();(function() {
    let view = definePackage("jtekma.view");

    view.label = class Italic extends view.main {

        constructor(content) {
            super("label");
            this.setContent(content);
        }

    };

})();(function() {
    let view = definePackage("jtekma.view");

    view.img = class Italic extends view.main {

        constructor(src, alt) {
            super("img");
            if (!isTextEmpty(src)) {
                this.setAttribute("src", src);
            }
            if (!isTextEmpty(alt)) {
                this.setAttribute("alt", alt);
            }
        }

        setSrc(src) {
            this.setAttribute("src", src);
        }

        setAlt(alt) {
            this.setAttribute("alt", alt);
        }

    };

})();