(function () {
    let restClient = definePackage("restClient");

    restClient.response = class RestResponse {

        constructor(status) {
            this.status = status;
        }

        isSuccessful() {
            if (isNull(this.status)) {
                return false;
            }
            return this.status >= 200 && this.status <= 299;
        }

        parseResponseHeaders(responseHeaders) {
            if (isTextEmpty(responseHeaders)) {
                return;
            }
            this.headers = {};
            for (let line of responseHeaders.split("\n")) {
                let separator = line.indexOf(":");
                if (separator < 0) {
                    continue;
                }
                let headerName = trimText(line.substring(0, separator));
                let headerValue = trimText(line.substring(separator + 1));
                this.headers[headerName] = headerValue;
            }
        }

        getContentType() {
            for (let headerName in this.headers) {
                if ("content-type" == headerName.toLowerCase()) {
                    return this.headers[headerName];
                }
            }
        }

        isTextResponse() {
            let responseContentType = this.getContentType();
            if (isNull(responseContentType)) {
                return false;
            }
            responseContentType = responseContentType.toLowerCase();
            return responseContentType.startsWith("text/") || responseContentType.startsWith("application/json") || responseContentType.startsWith("application/xml");
        }

        isJsonResponse() {
            let responseContentType = this.getContentType();
            if (isNull(responseContentType)) {
                return false;
            }
            responseContentType = responseContentType.toLowerCase();
            return responseContentType.startsWith("text/json") || responseContentType.startsWith("application/json");
        }

    };

    restClient.request = class RestRequest {

        constructor(method) {
            if (!isTextEmpty(method)) {
                this.method = method;
            }
            else {
                this.method = "GET";
            }
        }

        withParameters(parameters) {
            if (isNull(this.parameters)) {
                this.parameters = {};
            }
            for (let parameterName in parameters) {
                this.parameters[parameterName] = parameters[parameterName];
            }
            return this;
        }

        withParameter(parameterName, parameterValue) {
            if (isTextEmpty(parameterName)) {
                throw new Error("Parameter name not provided");
            }
            if (isNull(this.parameters)) {
                this.parameters = {};
            }
            this.parameters[parameterName] = parameterValue;
            return this;
        }

        withHeaders(headers) {
            if (isNull(this.headers)) {
                this.headers = {};
            }
            for (let headerName in headers) {
                this.headers[headerName] = headers[headerName];
            }
            return this;
        }

        withHeader(headerName, headerValue) {
            if (isTextEmpty(headerName)) {
                throw new Error("Header name not provided");
            }
            if (isNull(this.headers)) {
                this.headers = {};
            }
            this.headers[headerName] = headerValue;
            return this;
        }

        withBody(body) {
            this.body = body;
            return this;
        }

        withCache(cache) {
            this.cache = cache;
            return this;
        }

        getContentType() {
            for (let headerName in this.headers) {
                if ("content-type" == headerName.toLowerCase()) {
                    return this.headers[headerName];
                }
            }
        }

        execute(url, callback) {
            if (!isNull(callback) && !isFunction(callback)) {
                throw new Error("Callback parameter needs to be a function");
            }

            let targetUrl = contextPath + url;

            if (isNotNull(this.parameters)) {
                for (let parameterName in this.parameters) {
                    if (targetUrl.indexOf("?") >= 0) {
                        targetUrl += "&";
                    }
                    else {
                        targetUrl += "?";
                    }
                    targetUrl += parameterName + "=" + encodeURIComponent(this.parameters[parameterName]);
                }
            }

            let response = null;

            let httpRequest = new XMLHttpRequest();
            httpRequest.open(this.method, targetUrl, isNotNull(callback));
            if (isNotNull(this.headers)) {
                for (let headerName in this.headers) {
                    httpRequest.setRequestHeader(headerName, this.headers[headerName]);
                }
            }

            if (!this.cache) {
                httpRequest.setRequestHeader('Cache-Control', 'no-cache, no-store, max-age=0');
                httpRequest.setRequestHeader('Expires', 'Thu, 1 Jan 1970 00:00:00 GMT');
                httpRequest.setRequestHeader('Pragma', 'no-cache');
            }

            httpRequest.onreadystatechange = function() {
                if (this.readyState == 4) {
                    response = new restClient.response(this.status);
                    response.parseResponseHeaders(this.getAllResponseHeaders());

                    if (response.isTextResponse()) {
                        response.data = this.responseText;
                    }
                    else {
                        response.data = this.response;
                    }

                    if (this.status >= 200 && this.status <= 299) {
                        if (isNotNull(callback)) {
                            callback(response.data, response);
                        }
                    }
                    else {
                        response.errorText = this.responseText;
                        if (isNotNull(callback)) {
                            callback(response.data, response);
                        }
                    }
                }
            };

            if (isNotNull(this.body)) {
                httpRequest.send(this.body);
            }
            else {
                httpRequest.send();
            }

            if (isNull(callback)) {
                return response;
            }
        }

    };

    restClient.get = function(url, parameters, headers, callback) {
        if (!isNull(callback) && !isFunction(callback)) {
            throw new Error("Callback parameter needs to be a function");
        }
        let request = new restClient.request("GET");
        if (!isNull(parameters)) {
            request.withParameters(parameters);
        }
        if (!isNull(headers)) {
            request.withHeaders(headers);
        }
        if (isNull(callback)) {
            let response = request.execute(url);
            if (response.isSuccessful()) {
                return response.data;
            }
            else {
                throw Error("Invalid response code: " + response.status);
            }
        }
        else {
            request.execute(url, callback);
        }
    }

    restClient.getJson = function(url, parameters, headers, callback) {
        if (!isNull(callback) && !isFunction(callback)) {
            throw new Error("Callback parameter needs to be a function");
        }
        let request = new restClient.request("GET");
        if (!isNull(parameters)) {
            request.withParameters(parameters);
        }
        let allHeaders = {
            "Accept": "application/json"
        };
        if (!isNull(headers)) {
            for (let headerName in headers) {
                allHeaders[headerName] = headers[headerName];
            }
        }
        request.withHeaders(allHeaders);
        if (isNull(callback)) {
            let response = request.execute(url);
            if (response.isSuccessful()) {
                if (response.isJsonResponse())  {
                    response.data = JSON.parse(response.data);
                }
                return response.data;
            }
            else {
                throw Error("Invalid response code: " + response.status);
            }
        }
        else {
            request.execute(url, (data, response) => {
                if (response.isSuccessful() && response.isJsonResponse()) {
                    response.data = JSON.parse(response.data);
                }
                callback(response.data, response);
            });
        }
    }

    restClient.post = function(url, body, parameters, headers, callback) {
        if (!isNull(callback) && !isFunction(callback)) {
            throw new Error("Callback parameter needs to be a function");
        }
        let request = new restClient.request("POST");
        if (!isNull(parameters)) {
            request.withParameters(parameters);
        }
        if (!isNull(headers)) {
            request.withHeaders(headers);
        }
        if (!isNull(body)) {
            request.withBody(body);
        }
        if (isNull(callback)) {
            let response = request.execute(url);
            if (response.isSuccessful()) {
                return response.data;
            }
            else {
                throw Error("Invalid response code: " + response.status);
            }
        }
        else {
            request.execute(url, callback);
        }
    }

    restClient.postFormData = function(url, body, parameters, headers, callback) {
        if (!isNull(callback) && !isFunction(callback)) {
            throw new Error("Callback parameter needs to be a function");
        }
        let request = new restClient.request("POST");
        if (!isNull(parameters)) {
            request.withParameters(parameters);
        }
        if (!isNull(headers)) {
            request.withHeaders(headers);
        }
        request.withHeader("Content-Type", "multipart/form-data");
        if (!isNull(body)) {
            request.withBody(body);
        }
        if (isNull(callback)) {
            let response = request.execute(url);
            if (response.isSuccessful()) {
                return response.data;
            }
            else {
                throw Error("Invalid response code: " + response.status);
            }
        }
        else {
            request.execute(url, callback);
        }
    }

    restClient.postJson = function(url, body, parameters, headers, callback) {
        if (!isNull(callback) && !isFunction(callback)) {
            throw new Error("Callback parameter needs to be a function");
        }
        let request = new restClient.request("POST");
        if (!isNull(parameters)) {
            request.withParameters(parameters);
        }
        let allHeaders = {
            "Accept": "application/json"
        };
        if (!isNull(body)) {
            allHeaders["Content-Type"] = "application/json; charset=utf8";
        }
        if (!isNull(headers)) {
            for (let headerName in headers) {
                allHeaders[headerName] = headers[headerName];
            }
        }
        request.withHeaders(allHeaders);
        if (!isNull(body)) {
            request.withBody(JSON.stringify(body));
        }
        if (isNull(callback)) {
            let response = request.execute(url);
            if (response.isSuccessful()) {
                if (response.isJsonResponse())  {
                    response.data = JSON.parse(response.data);
                }
                return response.data;
            }
            else {
                throw Error("Invalid response code: " + response.status);
            }
        }
        else {
            request.execute(url, (data, response) => {
                if (response.isSuccessful() && response.isJsonResponse()) {
                    response.data = JSON.parse(response.data);
                }
                callback(response.data, response);
            });
        }
    }

    restClient.put = function(url, body, parameters, headers, callback) {
        if (!isNull(callback) && !isFunction(callback)) {
            throw new Error("Callback parameter needs to be a function");
        }
        let request = new restClient.request("PUT");
        if (!isNull(parameters)) {
            request.withParameters(parameters);
        }
        if (!isNull(headers)) {
            request.withHeaders(headers);
        }
        if (!isNull(body)) {
            request.withBody(body);
        }
        if (isNull(callback)) {
            let response = request.execute(url);
            if (response.isSuccessful()) {
                return response.data;
            }
            else {
                throw Error("Invalid response code: " + response.status);
            }
        }
        else {
            request.execute(url, callback);
        }
    }

    restClient.putJson = function(url, body, parameters, headers, callback) {
        if (!isNull(callback) && !isFunction(callback)) {
            throw new Error("Callback parameter needs to be a function");
        }
        let request = new restClient.request("PUT");
        if (!isNull(parameters)) {
            request.withParameters(parameters);
        }
        let allHeaders = {
            "Accept": "application/json"
        };
        if (!isNull(body)) {
            allHeaders["Content-Type"] = "application/json; charset=utf8";
        }
        if (!isNull(headers)) {
            for (let headerName in headers) {
                allHeaders[headerName] = headers[headerName];
            }
        }
        request.withHeaders(allHeaders);
        if (!isNull(body)) {
            request.withBody(JSON.stringify(body));
        }
        if (isNull(callback)) {
            let response = request.execute(url);
            if (response.isSuccessful()) {
                if (response.isJsonResponse())  {
                    response.data = JSON.parse(response.data);
                }
                return response.data;
            }
            else {
                throw Error("Invalid response code: " + response.status);
            }
        }
        else {
            request.execute(url, (data, response) => {
                if (response.isSuccessful() && response.isJsonResponse()) {
                    response.data = JSON.parse(response.data);
                }
                callback(response.data, response);
            });
        }
    }

    restClient.delete = function(url, parameters, headers, callback) {
        if (!isNull(callback) && !isFunction(callback)) {
            throw new Error("Callback parameter needs to be a function");
        }
        let request = new restClient.request("DELETE");
        if (!isNull(parameters)) {
            request.withParameters(parameters);
        }
        if (!isNull(headers)) {
            request.withHeaders(headers);
        }
        if (isNull(callback)) {
            let response = request.execute(url);
            if (response.isSuccessful()) {
                return response.data;
            }
            else {
                throw Error("Invalid response code: " + response.status);
            }
        }
        else {
            request.execute(url, callback);
        }
    }

})();
