(function () {
    var restClient = definePackage("restClient");

    restClient.response = class RestResponse {

        constructor(status) {
            this.status = status;
        }

        isSuccessful() {
            if (isNull(this.status)) {
                return false;
            }
            return this.status >= 200 && this.status <= 299;
        }

        parseResponseHeaders(responseHeaders) {
            if (isTextEmpty(responseHeaders)) {
                return;
            }
            this.headers = {};
            for (var line of responseHeaders.split("\n")) {
                var separator = line.indexOf(":");
                if (separator < 0) {
                    continue;
                }
                var headerName = trimText(line.substring(0, separator));
                var headerValue = trimText(line.substring(separator + 1));
                this.headers[headerName] = headerValue;
            }
        }

    };

    restClient.request = class RestRequest {

        constructor(method) {
            if (!isTextEmpty(method)) {
                this.method = method;
            }
            else {
                this.method = "GET";
            }
        }

        withParameters(parameters) {
            if (isNull(this.parameters)) {
                this.parameters = {};
            }
            for (var parameterName in parameters) {
                this.parameters[parameterName] = parameters[parameterName];
            }
            return this;
        }

        withParameter(parameterName, parameterValue) {
            if (isTextEmpty(parameterName)) {
                throw new Error("Parameter name not provided");
            }
            if (isNull(this.parameters)) {
                this.parameters = {};
            }
            this.parameters[parameterName] = parameterValue;
            return this;
        }

        withHeaders(headers) {
            if (isNull(this.headers)) {
                this.headers = {};
            }
            for (var headerName in headers) {
                this.headers[headerName] = headers[headerName];
            }
            return this;
        }

        withHeader(headerName, headerValue) {
            if (isTextEmpty(headerName)) {
                throw new Error("Header name not provided");
            }
            if (isNull(this.headers)) {
                this.headers = {};
            }
            this.headers[headerName] = headerValue;
            return this;
        }

        withBody(body) {
            this.body = body;
            return this;
        }

        withEncType(enctype) {
            this.enctype = enctype;
            return this;
        }

        execute(url, callback) {
            if (!isNull(callback) && !isFunction(callback)) {
                throw new Error("Callback parameter needs to be a function");
            }

            var targetUrl = contextPath + url;

            if (!isNull(this.parameters)) {
                for (var parameterName in this.parameters) {
                    if (targetUrl.indexOf("?") >= 0) {
                        targetUrl += "&";
                    }
                    else {
                        targetUrl += "?";
                    }
                    targetUrl += parameterName + "=" + encodeURIComponent(this.parameters[parameterName]);
                }
            }

            var request = {
                "url": targetUrl
            };
            request.type = this.method;

            if (!isNull(this.headers)) {
                request.headers = this.headers;
            }

            if (!isNull(this.enctype)) {
                request.enctype = this.enctype;
                if (request.enctype == "multipart/form-data") {
                    request.processData = false;
                    request.contentType = false;
                }
            }

            if (!isNull(this.body)) {
                request.data = this.body;
            }

            if (isNull(callback)) {
                request.async = false;
            }
            request.cache = false;

            var response = null;

            request.success = function(data, textStatus, jqXHR) {
                response = new restClient.response(jqXHR.status);
                response.parseResponseHeaders(jqXHR.getAllResponseHeaders());
                response.data = data;
                if (!isNull(callback)) {
                    callback(response.data, response);
                }
            }

            request.error = function(data) {
                response = new restClient.response(data.status);
                response.errorText = data.responseText;
                if (!isNull(callback)) {
                    callback(response.data, response);
                }
            }

            $.ajax(request);

            if (isNull(callback)) {
                return response;
            }
        }

    };

    restClient.get = function(url, parameters, headers, callback) {
        if (!isNull(callback) && !isFunction(callback)) {
            throw new Error("Callback parameter needs to be a function");
        }
        var request = new restClient.request("GET");
        if (!isNull(parameters)) {
            request.withParameters(parameters);
        }
        if (!isNull(headers)) {
            request.withHeaders(headers);
        }
        if (isNull(callback)) {
            var response = request.execute(url);
            if (response.isSuccessful()) {
                return response.data;
            }
            else {
                throw Error("Invalid response code: " + response.status);
            }
        }
        else {
            request.execute(url, callback);
        }
    }

    restClient.post = function(url, body, parameters, headers, callback) {
        if (!isNull(callback) && !isFunction(callback)) {
            throw new Error("Callback parameter needs to be a function");
        }
        var request = new restClient.request("POST");
        if (!isNull(parameters)) {
            request.withParameters(parameters);
        }
        if (!isNull(headers)) {
            request.withHeaders(headers);
        }
        if (!isNull(body)) {
            request.withBody(body);
        }
        if (isNull(callback)) {
            var response = request.execute(url);
            if (response.status >= 200 && response.status <= 299) {
                return response.data;
            }
            else {
                throw Error("Invalid response code: " + response.status);
            }
        }
        else {
            request.execute(url, callback);
        }
    }

    restClient.postFormData = function(url, body, parameters, headers, callback) {
        if (!isNull(callback) && !isFunction(callback)) {
            throw new Error("Callback parameter needs to be a function");
        }
        var request = new restClient.request("POST");
        if (!isNull(parameters)) {
            request.withParameters(parameters);
        }
        if (!isNull(headers)) {
            request.withHeaders(headers);
        }
        request.withEncType("multipart/form-data");
        if (!isNull(body)) {
            request.withBody(body);
        }
        if (isNull(callback)) {
            var response = request.execute(url);
            if (response.status >= 200 && response.status <= 299) {
                return response.data;
            }
            else {
                throw Error("Invalid response code: " + response.status);
            }
        }
        else {
            request.execute(url, callback);
        }
    }

    restClient.postJson = function(url, body, parameters, headers, callback) {
        if (!isNull(callback) && !isFunction(callback)) {
            throw new Error("Callback parameter needs to be a function");
        }
        var request = new restClient.request("POST");
        if (!isNull(parameters)) {
            request.withParameters(parameters);
        }
        var allHeaders = {
            "Accept": "application/json"
        };
        if (!isNull(body)) {
            allHeaders["Content-Type"] = "application/json; charset=utf8";
        }
        if (!isNull(headers)) {
            for (var headerName in headers) {
                allHeaders[headerName] = headers[headerName];
            }
        }
        request.withHeaders(allHeaders);
        if (!isNull(body)) {
            request.withBody(JSON.stringify(body));
        }
        if (isNull(callback)) {
            var response = request.execute(url);
            if (response.status >= 200 && response.status <= 299) {
                return response.data;
            }
            else {
                throw Error("Invalid response code: " + response.status);
            }
        }
        else {
            request.execute(url, callback);
        }
    }

    restClient.put = function(url, body, parameters, headers, callback) {
        if (!isNull(callback) && !isFunction(callback)) {
            throw new Error("Callback parameter needs to be a function");
        }
        var request = new restClient.request("PUT");
        if (!isNull(parameters)) {
            request.withParameters(parameters);
        }
        if (!isNull(headers)) {
            request.withHeaders(headers);
        }
        if (!isNull(body)) {
            request.withBody(body);
        }
        if (isNull(callback)) {
            var response = request.execute(url);
            if (response.status >= 200 && response.status <= 299) {
                return response.data;
            }
            else {
                throw Error("Invalid response code: " + response.status);
            }
        }
        else {
            request.execute(url, callback);
        }
    }

    restClient.putJson = function(url, body, parameters, headers, callback) {
        if (!isNull(callback) && !isFunction(callback)) {
            throw new Error("Callback parameter needs to be a function");
        }
        var request = new restClient.request("PUT");
        if (!isNull(parameters)) {
            request.withParameters(parameters);
        }
        var allHeaders = {
            "Accept": "application/json"
        };
        if (!isNull(body)) {
            allHeaders["Content-Type"] = "application/json; charset=utf8";
        }
        if (!isNull(headers)) {
            for (var headerName in headers) {
                allHeaders[headerName] = headers[headerName];
            }
        }
        request.withHeaders(allHeaders);
        if (!isNull(body)) {
            request.withBody(JSON.stringify(body));
        }
        if (isNull(callback)) {
            var response = request.execute(url);
            if (response.status >= 200 && response.status <= 299) {
                return response.data;
            }
            else {
                throw Error("Invalid response code: " + response.status);
            }
        }
        else {
            request.execute(url, callback);
        }
    }

    restClient.delete = function(url, parameters, headers, callback) {
        if (!isNull(callback) && !isFunction(callback)) {
            throw new Error("Callback parameter needs to be a function");
        }
        var request = new restClient.request("DELETE");
        if (!isNull(parameters)) {
            request.withParameters(parameters);
        }
        if (!isNull(headers)) {
            request.withHeaders(headers);
        }
        if (isNull(callback)) {
            var response = request.execute(url);
            if (response.status >= 200 && response.status <= 299) {
                return response.data;
            }
            else {
                throw Error("Invalid response code: " + response.status);
            }
        }
        else {
            request.execute(url, callback);
        }
    }

})();
