function isNull(obj) {
	return (obj === undefined || obj === null);
};

function isNotNull(obj) {
	return (obj !== undefined && obj !== null);
};

function isFunction(obj) {
	if (isNull(obj)) {
		return false;
	}
	var getType = {};
	return obj && getType.toString.call(obj) === '[object Function]';
};

function isArray(obj) {
	if (isNull(obj)) {
		return false;
	}
	return Object.prototype.toString.call(obj) === '[object Array]';
};

function isObject(obj) {
	if (isNull(obj)) {
		return false;
	}
	return (typeof obj === 'object');
};

function isString(obj) {
    if (isNull(obj)) {
		return false;
	}
	return (typeof obj === 'string');
}

function isNumber(obj) {
    if (isNull(obj)) {
		return false;
	}
	return (typeof obj === 'number');
}

function isValidNumber(val) {
	if (isTextEmpty(val)) {
		return false;
	}
	return val == Number(val);
};

function cloneObject(obj) {
    if (isNull(obj)) {
        return null;
    }
    if (isObject(obj)) {
        var copy = {};
        for (var key in obj) {
            copy[key] = obj[key];
        }
        return copy;
    }
    else if (isArray(obj)) {
        var copy = [];
        for (var key in obj) {
            copy[key] = obj[key];
        }
        return copy;
    }
    else {
    throw Error("Given object is not valid object type");
    }
};

function escapeHtml(text) {
	if (isNull(text)) {
		return null;
	}
	text = text.replace(new RegExp("&", 'g'), "&amp;");
	text = text.replace(new RegExp("<", 'g'), "&lt;");
	text = text.replace(new RegExp(">", 'g'), "&gt;");
	text = text.replace(new RegExp("\"", 'g'), "&quot;");
	text = text.replace(new RegExp("'", 'g'), "&#39;");
	text = text.replace(new RegExp("/", 'g'), "&#x2F;");
	return text;
};

function escapeRegExp(string) {
  return string.replace(/[.*+?^${}()|[\]\\]/g, "\\$&");
}


function isTextEmpty(text) {
	if (isNull(text)) {
		return true;
	}
	return text === "";
};

function trimText(text) {
	if (isTextEmpty(text)) {
		return "";
	}
	return String(text).trim();
};

function definePackage(packageName, target) {
    if (isTextEmpty(packageName)) {
        throw new Error("Package name is empty");
    }

    var packages = packageName.split(".");
    for (var subPackageName of packages) {
        if (isTextEmpty(subPackageName)) {
            throw new Error("Invalid package name: '" + packageName + "'");
        }
    }

    var currentPackage = target;
    if (isNull(currentPackage)) {
        currentPackage = window;
    }

    for (var subPackageName of packages) {
        if (isNull(currentPackage[subPackageName])) {
            currentPackage[subPackageName] = {};
            currentPackage[subPackageName].definePackage = function(name) {
                return definePackage(name, currentPackage[subPackageName]);
            };
        }
        currentPackage = currentPackage[subPackageName];
    }

    return currentPackage;
};

function defineModule(moduleName, module) {
    if (isTextEmpty(moduleName)) {
        throw new Error("Module name is empty");
    }
    if (isNull(module)) {
        throw new Error("Module not provided");
    }
    var packages = moduleName.split(".");
    if (packages.length == 1) {
        window[packages[0]] = module;
    }
    else {
        var pckg = definePackage(packages.slice(0, packages.length - 1).join("."));
        pckg[packages[packages.length - 1]] = module;
    }

    return module;
};

function loadModule(moduleName) {
    if (isTextEmpty(moduleName)) {
        throw new Error("Module name is empty");
    }

    var currentPackage = window;
    var packages = moduleName.split(".");
    for (var subPackageName of packages) {
        if (isTextEmpty(subPackageName)) {
            throw new Error("Invalid module name: '" + moduleName + "'");
        }
        if (isNull(currentPackage[subPackageName])) {
            throw new Error("Module '" + moduleName + "' not defined");
        }
        currentPackage = currentPackage[subPackageName];
    }

    if (isNull(currentPackage)) {
        throw new Error("Module '" + moduleName + "' not defined");
    }

    return currentPackage;
};
