(function() {
    definePackage("jtekma");
    let view = definePackage("jtekma.view");

    view.main = class MainView {

        constructor(tag, template, model, attributes, events) {
            if (isTextEmpty(tag)) {
                throw new Error("Tag needs to be provided");
            }
            this.el = document.createElement(tag);
            this.tag = tag;

            if (isNotNull(template) && !isString(template) && !isFunction(template)) {
                throw new Error("Template can only be string or function");
            }
            else {
                this.template = template;
            }

            if (!isNull(model)) {
                this.model = model;
            }
            else {
                this.model = {};
            }
            if (isNotNull(attributes)) {
                this.attributes = attributes;
            }
            else {
                this.attributes = {};
            }
            if (isNotNull(events)) {
                this.events = events;
            }
            else {
                this.events = {};
            }
        }

        setContent(content) {
            if (isString(content)) {
                this.setContentText(content);
            }
            else {
                this.content = content;
            }
        }

        setContentText(content) {
            if (!isTextEmpty(content)) {
                this.content = escapeHtml(content);
            }
            else {
                this.content = "";
            }
        }

        setContentHtml(content) {
            if (!isTextEmpty(content)) {
                this.content = content;
            }
            else {
                this.content = "";
            }
        }

        addContent(item) {
            if (isNull(item)) {
                return;
            }
            if (isNull(this.content)) {
                this.content = [];
            }
            if (!isArray(this.content)) {
                let c = this.content;
                this.content = [c];
            }
            if (isArray(item)) {
                for (let i of item) {
                    this.content.push(i);
                }
            }
            else {
                this.content.push(item);
            }
        }

        setId(id) {
            this.setAttribute("id", id);
        }

        setClass(clazz) {
            this.setAttribute("class", clazz);
        }

        addClass(clazz) {
            if (isTextEmpty(this.attributes.class)) {
                this.setAttribute("class", clazz);
            }
            else {
                let updatedClass = "";
                let classAlreadyPresent = false;
                for (let p of this.attributes.class.split(" ")) {
                    if (isTextEmpty(p)) {
                        continue;
                    }
                    if (p == clazz) {
                        classAlreadyPresent = true;
                    }
                    if (updatedClass.length > 0) {
                        updatedClass += " ";
                    }
                    updatedClass += p;
                }
                if (!classAlreadyPresent) {
                    if (updatedClass.length > 0) {
                        updatedClass += " ";
                    }
                    updatedClass += clazz;
                }
                this.setAttribute("class", updatedClass);
            }
        }

        removeClass(clazz) {
            if (isTextEmpty(this.attributes.class)) {
                return;
            }
            let updatedClass = "";
            for (let p of this.attributes.class.split(" ")) {
                if (isTextEmpty(p)) {
                    continue;
                }
                if (p == clazz) {
                    continue;
                }
                if (updatedClass.length > 0) {
                    updatedClass += " ";
                }
                updatedClass += p;
            }
            this.setAttribute("class", updatedClass);
        }

        setStyle(style) {
            this.setAttribute("style", style);
        }

        addStyle(style, value) {
            let styleKey = style;
            if (isNull(value)) {
                let parts = style.split(":");
                if (parts.length > 1) {
                    styleKey = parts[0];
                    value = parts[1];
                }
            }
            if (isTextEmpty(this.attributes.style)) {
                this.setAttribute("style", styleKey + ": " + value);
            }
            else {
                let updatedStyle = "";
                let parts = this.attributes.style.split(";");
                let styleOverridden = false;
                for (let p of parts) {
                    if (isTextEmpty(p)) {
                        continue;
                    }
                    let t = p.split(":");
                    if (t.length > 1) {
                        if (t[0].trim() == styleKey.trim()) {
                            styleOverridden = true;
                            updatedStyle += styleKey.trim() + ": " + value.trim() + ";";
                        }
                        else {
                            updatedStyle += p + ";";
                        }
                    }
                    else {
                        updatedStyle += p + ";";
                    }
                }
                if (!styleOverridden) {
                    updatedStyle += styleKey.trim() + ": " + value.trim() + ";";
                }
                this.setAttribute("style", updatedStyle);
            }
            if (!this.attributes.style.endsWith(";")) {
                this.setAttribute("style", this.attributes.style + ";");
            }
        }

        removeStyle(style) {
            if (isTextEmpty(this.attributes.style)) {
                return;
            }
            let updatedStyle = "";
            for (let p of this.attributes.style.split(";")) {
                if (isTextEmpty(p)) {
                    continue;
                }
                let t = p.split(":");
                if (t.length > 1) {
                    if (t[0].trim() != style.trim()) {
                        updatedStyle += p + ";";
                    }
                }
                else {
                    updatedStyle += p + ";";
                }
            }
            this.setAttribute("style", updatedStyle);
        }

        setAttribute(key, value) {
            if (isNull(this.attributes)) {
                this.attributes = {};
            }
            this.attributes[key] = value;
        }

        removeAttribute(key) {
            if (isNotNull(this.attributes)) {
                this.attributes[key] = null;
            }
        }

        onClick(callback) {
            this.addEventListener("click", callback);
        }

        addEventListener(eventType, callback) {
            if (!isFunction(callback)) {
                throw new Error("Callback parameter needs to be a function");
            }
            this.events[eventType] = callback;
        }

        show() {
            this.removeStyle("display");
            this.draw();
        }

        hide() {
            this.addStyle("display", "none");
            this.draw();
        }

        render() {
            if (isNull(this.template)) {
                return null;
            }
            if (isFunction(this.preRender)) {
                this.preRender();
            }
            let t = null;
            if (isString(this.template)) {
                t = this.template;
            }
            else if (isFunction(this.template)) {
                t = this.template();
            }
            else {
                throw new Error("Template is not string or function");
            }
            if (isNull(t)) {
                return null;
            }
            let html = Mustache.render(t, this.model);
            if (isFunction(this.postRender)) {
                this.postRender();
            }
            return html;
        }

        draw(target) {
            let html = this.render();
            if (isFunction(this.preDraw)) {
                this.preDraw();
            }
            if (isNotNull(html)) {
                this.el.innerHTML = html;
            }
            else {
                if (isNull(this.content)) {
                    this.el.innerHTML = "";
                }
                else if (isString(this.content)) {
                    this.el.innerHTML = this.content;
                }
                else if (isArray(this.content)) {
                    this.el.innerHTML = "";
                    for (let item of this.content) {
                        if (isNull(item)) {
                            // ignore item
                        }
                        else if (isString(item)) {
                            this.el.append(item);
                        }
                        else if (isObject(item) && isFunction(item.draw)) {
                            item.draw(this.el);
                        }
                        else {
                            throw new Error("Content item for drawing is not supported");
                        }
                    }
                }
                else if (isObject(this.content) && isFunction(this.content.draw)) {
                    this.el.innerHTML = "";
                    this.content.draw(this.el);
                }
                else {
                    throw new Error("Content for drawing is not supported");
                }
            }
            if (isNotNull(this.attributes)) {
                for (let attributeKey in this.attributes) {
                    if (isNotNull(this.attributes[attributeKey])) {
                        this.el.setAttribute(attributeKey, this.attributes[attributeKey]);
                    }
                    else {
                        this.el.removeAttribute(attributeKey);
                    }
                }
            }
            if (isNotNull(this.events)) {
                for (let eventKey in this.events) {
                    if (isNull(this.events[eventKey])) {
                        continue;
                    }
                    if (!isFunction(this.events[eventKey])) {
                        throw new Error("Event for type '" + eventKey + "' is not function");
                    }
                    this.el.addEventListener(eventKey, this.events[eventKey]);
                }
            }
            if (isNotNull(target)) {
                target.append(this.el);
            }
            if (isFunction(this.postDraw)) {
                this.postDraw();
            }
        }
    };

})();